<?php

/**
 * Common PayKeeper payment system PHP class for use in PayKeeper payment modules for different CMS.
 */
class PaykeeperPayment
{
    /**
     * @var array Fiscal (fz54) cart
     */
    public $fiscal_cart = [];

    /**
     * @var float Order total sum
     */
    public $order_total = 0;

    /**
     * @var float Shipping price
     */
    public $shipping_price = 0;

    /**
     * @var bool Flag indicating whether to use taxes
     */
    public $use_taxes = false;

    /**
     * @var bool Flag indicating whether to use delivery
     */
    public $use_delivery = false;

    /**
     * @var int Index of the delivery item in the cart
     */
    public $delivery_index = -1;

    /**
     * @var int Index of the single item in the cart
     */
    public $single_item_index = -1;

    /**
     * @var int Index of the item with quantity greater than one in the cart
     */
    public $more_than_one_item_index = -1;

    /**
     * @var array|null Order parameters
     */
    public $order_params = null;

    /**
     * @var array Discounts
     */
    public $discounts = [];

    /**
     * Sets order parameters.
     *
     * @param float $order_total
     * @param string $clientid
     * @param string $orderid
     * @param string $client_email
     * @param string $client_phone
     * @param string $service_name
     * @param string $form_url
     * @param string $secret_key
     */
    public function setOrderParams(
        $order_total = 0,
        $clientid = "",
        $orderid = "",
        $client_email = "",
        $client_phone = "",
        $service_name = "",
        $form_url = "",
        $secret_key = ""
    ) {
        $this->setOrderTotal($order_total);
        $this->order_params = [
            "sum" => $order_total,
            "clientid" => $clientid,
            "orderid" => $orderid,
            "client_email" => $client_email,
            "client_phone" => $client_phone,
            "phone" => $client_phone,
            "service_name" => $service_name,
            "form_url" => $form_url,
            "secret_key" => $secret_key,
        ];
    }

    /**
     * Gets a specific order parameter.
     *
     * @param string $value
     * @return mixed
     */
    public function getOrderParams($value)
    {
        return array_key_exists($value, $this->order_params) ? $this->order_params[$value] : false;
    }

    /**
     * Updates the fiscal cart.
     *
     * @param string $ftype
     * @param string $name
     * @param float $price
     * @param int $quantity
     * @param float $sum
     * @param string $tax
     */
    public function updateFiscalCart($ftype, $name = "", $price = 0, $quantity = 0, $sum = 0, $tax = "none")
    {
        if ($ftype === "create") {
            $name = str_replace(["\n", "\r"], "", $name);
        }
        $price_to_add = number_format($price, 2, ".", "");
        $sum_to_add = number_format($price_to_add * $quantity, 2, ".", "");
        $this->fiscal_cart[] = [
            "name" => $name,
            "price" => $price_to_add,
            "quantity" => $quantity,
            "sum" => $sum_to_add,
            "tax" => $tax,
        ];
    }

    /**
     * Gets the fiscal cart.
     *
     * @return array
     */
    public function getFiscalCart()
    {
        return $this->fiscal_cart;
    }

    /**
     * Sets discounts.
     *
     * @param bool $discount_enabled_flag
     */
    public function setDiscounts($discount_enabled_flag)
    {
        $discount_modifier_value = 1;
        $shipping_included = false;

        if ($discount_enabled_flag && $this->getFiscalCartSum(false) > 0) {
            if ($this->getOrderTotal() >= $this->getShippingPrice()) {
                if ($this->getFiscalCartSum(false) > 0) {
                    $discount_modifier_value = ($this->getOrderTotal() - $this->getShippingPrice()) / $this->getFiscalCartSum(false);
                }
            } else {
                if ($this->getFiscalCartSum(true) > 0) {
                    $discount_modifier_value = $this->getOrderTotal() / $this->getFiscalCartSum(true);
                    $shipping_included = true;
                }
            }

            if ($discount_modifier_value < 1) {
                for ($pos = 0; $pos < count($this->getFiscalCart()); $pos++) {
                    if (!$shipping_included && $pos == $this->delivery_index) {
                        continue;
                    }
                    if ($this->fiscal_cart[$pos]["quantity"] > 0) {
                        $price = $this->fiscal_cart[$pos]["price"] * $discount_modifier_value;
                        $this->fiscal_cart[$pos]["price"] = number_format($price, 2, ".", "");
                        $sum = $this->fiscal_cart[$pos]["price"] * $this->fiscal_cart[$pos]["quantity"];
                        $this->fiscal_cart[$pos]["sum"] = number_format($sum, 2, ".", "");
                    }
                }
            }
        }
    }

    /**
     * Corrects precision issues in the fiscal cart.
     */
    public function correctPrecision()
    {
        $fiscal_cart_sum = $this->getFiscalCartSum(true);
        $total_sum = $this->getOrderTotal();
        $diff_value = $total_sum - $fiscal_cart_sum;

        if (abs($diff_value) >= 0.005) {
            $diff_sum = number_format($diff_value, 2, ".", "");
            if ($this->getUseDelivery()) {
                $this->correctPriceOfCartItem($diff_sum, count($this->fiscal_cart) - 1);
            } elseif ($this->single_item_index >= 0) {
                $this->correctPriceOfCartItem($diff_sum, $this->single_item_index);
            } elseif ($this->more_than_one_item_index >= 0) {
                $this->splitCartItem($this->more_than_one_item_index);
                $this->correctPriceOfCartItem($diff_sum, count($this->fiscal_cart) - 1);
            } else {
                $modify_value = ($diff_sum > 0) ? $total_sum / $fiscal_cart_sum : $fiscal_cart_sum / $total_sum;
                for ($pos = 0; $pos < count($this->getFiscalCart()); $pos++) {
                    if ($this->fiscal_cart[$pos]["quantity"] > 0) {
                        $sum = $this->fiscal_cart[$pos]["sum"] * $modify_value;
                        $this->fiscal_cart[$pos]["sum"] = number_format($sum, 2, ".", "");
                        $price = $this->fiscal_cart[$pos]["sum"] / $this->fiscal_cart[$pos]["quantity"];
                        $this->fiscal_cart[$pos]["price"] = number_format($price, 2, ".", "");
                    }
                }
            }
        }
    }

    /**
     * Sets the order total.
     *
     * @param float $value
     */
    public function setOrderTotal($value)
    {
        $this->order_total = $value;
    }

    /**
     * Gets the order total.
     *
     * @return float
     */
    public function getOrderTotal()
    {
        return $this->order_total;
    }

    /**
     * Sets the shipping price.
     *
     * @param float $value
     */
    public function setShippingPrice($value)
    {
        $this->shipping_price = $value;
    }

    /**
     * Gets the shipping price.
     *
     * @return float
     */
    public function getShippingPrice()
    {
        return $this->shipping_price;
    }

    /**
     * Gets the payment form type.
     *
     * @return string
     */
    public function getPaymentFormType()
    {
        return strpos($this->order_params["form_url"], "/order/inline") === true ? "order" : "create";
    }

    /**
     * Sets the use taxes flag.
     */
    public function setUseTaxes()
    {
        $this->use_taxes = true;
    }

    /**
     * Gets the use taxes flag.
     *
     * @return bool
     */
    public function getUseTaxes()
    {
        return $this->use_taxes;
    }

    /**
     * Sets the use delivery flag.
     */
    public function setUseDelivery()
    {
        $this->use_delivery = true;
    }

    /**
     * Gets the use delivery flag.
     *
     * @return bool
     */
    public function getUseDelivery()
    {
        return $this->use_delivery;
    }

    /**
     * Sets taxes.
     *
     * @param float $tax_rate
     * @param bool $zero_value_as_none
     * @return array
     */
    public function setTaxes($tax_rate, $zero_value_as_none = true)
    {
        $taxes = ["tax" => "none", "tax_sum" => 0];
        switch (number_format(floatval($tax_rate), 0, ".", "")) {
            case 0:
                if (!$zero_value_as_none) {
                    $taxes["tax"] = "vat0";
                }
                break;
            case 10:
                $taxes["tax"] = "vat10";
                break;
            case 18:
                $taxes["tax"] = "vat18";
                break;
            case 20:
                $taxes["tax"] = "vat20";
                break;
        }
        return $taxes;
    }

    /**
     * Checks if delivery is included in the fiscal cart.
     *
     * @param float $delivery_price
     * @param string $delivery_name
     * @return bool
     */
    public function checkDeliveryIncluded($delivery_price, $delivery_name)
    {
        foreach ($this->getFiscalCart() as $index => $item) {
            if ($item["name"] === $delivery_name && $item["price"] == $delivery_price && $item["quantity"] == 1) {
                $this->delivery_index = $index;
                return true;
            }
        }
        return false;
    }

    /**
     * Gets the sum of the fiscal cart.
     *
     * @param bool $delivery_included
     * @return string
     */
    public function getFiscalCartSum($delivery_included)
    {
        $fiscal_cart_sum = 0;
        foreach ($this->getFiscalCart() as $index => $item) {
            if (!$delivery_included && $index == $this->delivery_index) {
                continue;
            }
            $fiscal_cart_sum += $item["price"] * $item["quantity"];
        }
        return number_format($fiscal_cart_sum, 2, ".", "");
    }

    /**
     * Displays debug information.
     *
     * @param mixed $obj_to_debug
     */
    public function showDebugInfo($obj_to_debug)
    {
        echo "<pre>";
        var_dump($obj_to_debug);
        echo "</pre>";
    }

    /**
     * Corrects the price of a cart item.
     *
     * @param float $corr_price_to_add
     * @param int $item_position
     */
    public function correctPriceOfCartItem($corr_price_to_add, $item_position)
    {
        $this->fiscal_cart[$item_position]["price"] += $corr_price_to_add;
        $this->fiscal_cart[$item_position]["sum"] = number_format(
            $this->fiscal_cart[$item_position]["price"] * $this->fiscal_cart[$item_position]["quantity"],
            2,
            ".",
            ""
        );
    }

    /**
     * Splits a cart item.
     *
     * @param int $cart_item_position
     */
    public function splitCartItem($cart_item_position)
    {
        $item_price = $this->fiscal_cart[$cart_item_position]["price"];
        $item_quantity = $this->fiscal_cart[$cart_item_position]["quantity"] - 1;

        $this->fiscal_cart[$cart_item_position]["quantity"] = $item_quantity;
        $this->fiscal_cart[$cart_item_position]["sum"] = $item_price * $item_quantity;

        $this->updateFiscalCart(
            $this->getPaymentFormType(),
            $this->fiscal_cart[$cart_item_position]["name"],
            $item_price,
            1,
            $item_price,
            $this->fiscal_cart[$cart_item_position]["tax"]
        );
    }

    /**
     * Gets the encoded fiscal cart.
     *
     * @return string
     */
    public function getFiscalCartEncoded()
    {
        return json_encode($this->getFiscalCart());
    }

    /**
     * Gets the default payment form.
     *
     * @param string $payment_form_sign
     * @return string
     */
    public function getDefaultPaymentForm($payment_form_sign)
    {
        $form = "";

        if ($this->getPaymentFormType() === "create") {
            $form = '
                <h3>Сейчас Вы будете перенаправлены на страницу банка.</h3>
                <form name="pay_form" id="pay_form" action="' . $this->getOrderParams("form_url") . '" accept-charset="utf-8" method="post">
                    <input type="hidden" name="sum" value="' . $this->getOrderTotal() . '"/>
                    <input type="hidden" name="orderid" value="' . $this->getOrderParams("orderid") . '"/>
                    <input type="hidden" name="clientid" value="' . $this->getOrderParams("clientid") . '"/>
                    <input type="hidden" name="client_email" value="' . $this->getOrderParams("client_email") . '"/>
                    <input type="hidden" name="client_phone" value="' . $this->getOrderParams("client_phone") . '"/>
                    <input type="hidden" name="service_name" value="' . $this->getOrderParams("service_name") . '"/>
                    <input type="hidden" name="cart" value="' . htmlentities($this->getFiscalCartEncoded(), ENT_QUOTES) . '"/>
                    <input type="hidden" name="sign" value="' . $payment_form_sign . '"/>
                    <input type="submit" class="btn btn-default" value="Оплатить"/>
                </form>
                <script type="text/javascript">
                    window.onload = function() {
                        setTimeout(fSubmit, 2000);
                    };
                    function fSubmit() {
                        document.forms["pay_form"].submit();
                    }
                </script>';
        } else {
            $payment_parameters = [
                "clientid" => $this->getOrderParams("clientid"),
                "orderid" => $this->getOrderParams('orderid'),
                "sum" => $this->getOrderTotal(),
                "client_phone" => $this->getOrderParams("phone"),
                "phone" => $this->getOrderParams("phone"),
                "client_email" => $this->getOrderParams("client_email"),
                "cart" => $this->getFiscalCartEncoded(),
            ];

            $query = http_build_query($payment_parameters);
            $err_text = null;

            if (function_exists("curl_init")) {
                $CR = curl_init();
                curl_setopt($CR, CURLOPT_URL, $this->getOrderParams("form_url"));
                curl_setopt($CR, CURLOPT_POST, 1);
                curl_setopt($CR, CURLOPT_FAILONERROR, true);
                curl_setopt($CR, CURLOPT_POSTFIELDS, $query);
                curl_setopt($CR, CURLOPT_RETURNTRANSFER, 1);
                curl_setopt($CR, CURLOPT_SSL_VERIFYPEER, 0);
                $result = curl_exec($CR);
                $error = curl_error($CR);

                if (!empty($error)) {
                    $form = "<br/><span class='message'>INTERNAL ERROR: $error</span>";
                    return false;
                } else {
                    $form = $result;
                }

                curl_close($CR);
            } else {
                if (!ini_get('allow_url_fopen')) {
                    $form = "<br/><span class='message'>INTERNAL ERROR: Option allow_url_fopen is not set in php.ini</span>";
                } else {
                    $query_options = [
                        "https" => [
                            "method" => "POST",
                            "header" => "Content-type: application/x-www-form-urlencoded",
                            "content" => $query,
                        ]
                    ];

                    $context = stream_context_create($query_options);
                    $form = file_get_contents($this->getOrderParams("form_url"), false, $context);
                }
            }
        }

        if (empty($form)) {
            $form = '<h3>Произошла ошибка при инициализации платежа</h3><p>' . htmlspecialchars($err_text) . '</p>';
        }

        return $form;
    }
}
