<?php
/*
* @package      PayKeeper payment plugin for j2store

* @subpackage   J2Store
* @author       Michael Kochkin
* @copyright    Copyright (c) 2019 PayKeeper company. All rights reserved.
* --------------------------------------------------------------------------------
*/

/** ensure this file is being included by a parent file */
defined('_JEXEC') or die('Restricted access');
require_once (JPATH_ADMINISTRATOR.'/components/com_j2store/library/plugins/payment.php');
require_once('paykeeper_common_class/paykeeper.class.php');
class plgJ2StorePayment_paykeeper extends J2StorePaymentPlugin
{
    /**
     * @var $_element  string  Should always correspond with the plugin's filename,
     *                         forcing it to be unique
     */
    var $_element    = 'payment_paykeeper';

    /**
     * Constructor
     *
     * @param object $subject The object to observe
     * @param   array  $config  An array that holds the plugin configuration
     * @since 1.5
     */
    function __construct(& $subject, $config)
    {
        parent::__construct($subject, $config);
        $this->loadLanguage( '', JPATH_ADMINISTRATOR );
    }

    /**
     * For compatibility of earlier versions
     *
     * @param $element  string      a valid payment plugin element
     * @return bool
     * @throws Exception
     */
    function onJ2StoreIsJ2Store4($element)
    {
        if (!$this->_isMe($element)) {
            return null;
        }
        return true;
    }

    /**
     * Wrapper for the internal _postPayment method
     * that processes the payment after user submits
     *
     * @param $element  string      a valid payment plugin element
     * @param $data     array       form post data
     * @return string
     * @throws Exception
     */
    function onJ2StorePostPayment($element, $data)
    {
        if (!$this->_isMe($element)) {
            return null;
        }
        return $this->_postPayment($data);
    }

    /**
     * Wrapper for the internal _postPayment method (callback)
     * that processes the payment after user submits
     *
     * @param $element  string      a valid payment plugin element
     * @param $data     array       form post data
     * @return string
     * @throws Exception
     */
    function onJ2StoreCallback($element, $data)
    {
        if (!$this->_isMe($element)) {
            return null;
        }
        return $this->_postPayment($data);
    }

    /**
     * Prepares variables for the payment form.
     * Displayed when the customer selects the method in the Shipping and Payment step of Checkout.
     *
     * @param array $data Form post data.
     * @return string
     */
    function _renderForm( $data )
    {
        $vars = new JObject();
        $vars->message = $this->params->get('description', '');
        $html = $this->_getLayout('message', $vars);

        //get the thank you message from the article (ID) provided in the plugin params
        $html .= $this->_displayArticle();

        return $html;
    }

    /**
     * Method to display a Place order button either to redirect the customer or process the credit card information.
     * @param $data     array       form post data
     * @return string   HTML to display
     */
    function _prePayment( $data )
    {
        // get component params
        $params = J2Store::config();
        $currency = J2Store::currency();

        // prepare the payment form
        F0FTable::addIncludePath(JPATH_ADMINISTRATOR.'/components/com_j2store/tables');
        $this->order = F0FTable::getInstance('Order', 'J2StoreTable')->getClone();
        $this->order->load(array('order_id'=>$data['order_id']));
        $orderinfo = $this->order->getOrderInformation();
        $app = JFactory::getApplication();

        $pk_obj = new PaykeeperPayment();

        //set order parameters
        $pk_obj->setOrderParams($this->order->order_total,  //sum
            $orderinfo->billing_first_name
            . " " .
            $orderinfo->billing_last_name,                  //clientid
            $this->order->order_id,                         //orderid
            $this->order->user_email,                       //client_email
            "",                                  //client_phone
            "",                                 //service_name
            $this->params->get('server', ''),               //payment form url
            $this->params->get('secret', '')                //secret key
        );

        //GENERATE FZ54 CART
        $cart_data = $this->order->get_line_items();
        $item_index = 0;
        foreach ($cart_data as $cart_item) {
            $taxes = array("tax" => "none", "tax_sum" => 0);
            $name = $cart_item["name"];
            $quantity = $cart_item["quantity"];
            $price = $cart_item["amount"];
            $sum = number_format($price*$quantity, 2, ".", "");
            if ($quantity == 1 && $pk_obj->single_item_index < 0)
                $pk_obj->single_item_index = $item_index;
            if ($quantity > 1 && $pk_obj->more_then_one_item_index < 0)
                $pk_obj->more_then_one_item_index = $item_index;
            if ($cart_item["tax"] != "none" && $price) {
                $tax_rate = $this->getTaxRate($cart_item["tax"], $price, FALSE);
                $taxes = $pk_obj->setTaxes($tax_rate);
            }
            $pk_obj->updateFiscalCart($pk_obj->getPaymentFormType(),
                $name, $price, $quantity, $sum, $taxes["tax"]);
            $item_index++;
        }

        //add shipping parameters to cart
        $shipping_tax_rate = 0;
        $shipping_taxes = array("tax" => "none", "tax_sum" => 0);
        $shipping_code = $app->input->getString("shipping_code");
        $shipping_price = (float) $app->input->getString("shipping_price");
        $shipping_extra = (float) $app->input->getString("shipping_extra");
        if ($shipping_code != ""){
            array_pop($pk_obj->fiscal_cart);
        }
        if ($app->input->getString("shipping_tax") != "none") {
            $shipping_tax = (float) $app->input->getString("shipping_tax");
            $pk_obj->setShippingPrice($shipping_price+$shipping_extra+$shipping_tax);

            if($shipping_price + $shipping_extra)
                $shipping_tax_rate = $this->getTaxRate($app->input->getString("shipping_tax"),($shipping_price+$shipping_extra), TRUE);

            $shipping_taxes = $pk_obj->setTaxes($shipping_tax_rate);
        }
        else
            $pk_obj->setShippingPrice($shipping_price+$shipping_extra);

        $shipping_name = $app->input->getString("shipping_name");
        if ($pk_obj->getShippingPrice() > 0) {
            $pk_obj->setUseDelivery(); //for precision correct check
            $pk_obj->updateFiscalCart($pk_obj->getPaymentFormType(),
                $shipping_name,
                $pk_obj->getShippingPrice(),
                1,
                $pk_obj->getShippingPrice(),
                $shipping_taxes["tax"]);
        }

        //set discounts
        $force_discounts_check = $this->params->get('force_discounts_check', '');
        $pk_obj->setDiscounts($force_discounts_check == 1);

        //handle possible precision problem
        $pk_obj->correctPrecision();

        $fiscal_cart_encoded = json_encode($pk_obj->getFiscalCart());

        //generate payment form
        $form = "";
        if ($pk_obj->getPaymentFormType() == "create") { //create form
            $to_hash = $pk_obj->getOrderTotal(TRUE)                .
                $pk_obj->getOrderParams("clientid")     .
                $pk_obj->getOrderParams("orderid")      .
                $pk_obj->getOrderParams("service_name") .
                $pk_obj->getOrderParams("client_email") .
                $pk_obj->getOrderParams("client_phone") .
                $pk_obj->getOrderParams("secret_key");
            $sign = hash ('sha256' , $to_hash);
            $form = '
                <h3>Нажмите на кнопку "Оплатить" для перенаправления на страницу банка.</h3>
                <form name="payment" id="pay_form" action="'.$pk_obj->getOrderParams("form_url").'" accept-charset="utf-8" method="post">
                <input type="hidden" name="sum" value = "'.$pk_obj->getOrderTotal(TRUE).'"/>
                <input type="hidden" name="orderid" value = "'.$pk_obj->getOrderParams("orderid").'"/>
                <input type="hidden" name="clientid" value = "'.$pk_obj->getOrderParams("clientid").'"/>
                <input type="hidden" name="client_email" value = "'.$pk_obj->getOrderParams("client_email").'"/>
                <input type="hidden" name="client_phone" value = "'.$pk_obj->getOrderParams("client_phone").'"/>
                <input type="hidden" name="service_name" value = "'.$pk_obj->getOrderParams("service_name").'"/>
                <input type="hidden" name="cart" value = \''.htmlentities($fiscal_cart_encoded,ENT_QUOTES).'\' />
                <input type="hidden" name="sign" value = "'.$sign.'"/>
                <input type="submit" class="button btn btn-primary" value="Оплатить"/>
                </form>';
        }
        else { //order form
            $payment_parameters = array(
                "clientid"=>$pk_obj->getOrderParams("clientid"),
                "orderid"=>$pk_obj->getOrderParams('orderid'),
                "sum"=>$pk_obj->getOrderTotal(TRUE),
                "phone"=>$pk_obj->getOrderParams("client_phone"),
                "client_phone"=>$pk_obj->getOrderParams("client_phone"),
                "client_email"=>$pk_obj->getOrderParams("client_email"),
                "cart"=>$fiscal_cart_encoded);
            $query = http_build_query($payment_parameters);
            $options = array("http"=>array(
                "method"=>"POST",
                "header"=>"Content-type: application/x-www-form-urlencoded",
                "content"=>$query
            ));
            $context = stream_context_create($options);
            if( function_exists( "curl_init" )) { //using curl
                $CR = curl_init();
                curl_setopt($CR, CURLOPT_URL, $pk_obj->getOrderParams("form_url"));
                curl_setopt($CR, CURLOPT_POST, 1);
                curl_setopt($CR, CURLOPT_FAILONERROR, TRUE);
                curl_setopt($CR, CURLOPT_POSTFIELDS, $query);
                curl_setopt($CR, CURLOPT_RETURNTRANSFER, 1);
                curl_setopt($CR, CURLOPT_SSL_VERIFYPEER, 0);
                $result = curl_exec( $CR );
                $error = curl_error( $CR );
                if( !empty( $error )) {
                    $form = "<br/><span class=message>"."INTERNAL ERROR:".$error."</span>";
                }
                else {
                    $form = $result;
                }
                curl_close($CR);
            }
            else { //using file_get_contents
                if (!ini_get('allow_url_fopen')) {
                    $form = "<br/><span class=message>"."INTERNAL ERROR: Option allow_url_fopen is not set in php.ini"."</span>";
                }
                else {
                    $form = file_get_contents($pk_obj->getOrderParams("form_url"), false, $context);
                }
            }
        }
        if ($form  == "") {
            $form = '<h3>Произошла ошибка при инциализации платежа</h3>';
        }

        $vars = new JObject();
        $vars->form = $form;
        return $this->_getLayout('prepayment', $vars);
    }

    /**
     * Processes the payment form
     * and returns HTML to be displayed to the user
     * generally with a success/failed message
     *
     * @param $data     array       form post data
     * @return string   HTML to display
     */
    function _postPayment( $data )
    {
        // Process the payment
        $app = JFactory::getApplication();
        $paction = $app->input->getString('paction');

        $vars = new JObject();

        switch ($paction)
        {
            case "callback":
                //process the response from the gateway
                echo $this->_processSale();
                $app->close();
                break;
            default :
                $vars->message = JText::_($this->params->get('onerrorpayment', ''));
                $html = $this->_getLayout('message', $vars);
                break;
        }

        return isset($html) ? $html : null;
    }


    /**
     * Processes the sale payment
     *
     */
    private function _processSale()
    {
        $app = JFactory::getApplication();
        $data = $app->input->getArray($_POST);

        //load the orderpayment record and set some values
        $order = F0FTable::getInstance( 'Order', 'J2StoreTable' )->getClone();

        if(empty($data['id']) || empty($data['orderid']))
            return "Error: empty orderid or id!";

        $secret = $this->params->get('secret', '');
        $id = $data['id'];
        $sum = $data['sum'];
        $clientid = $data['clientid'];
        $orderid = $data['orderid'];
        $key = $data['key'];

        if ($key != md5($id.number_format($sum, 2, '.', '').$clientid.$orderid.$secret))
            return "Error! Hash mismatch";

        if (!$order->load( array('order_id' => $orderid) ))
            return "Error! Incorrect order id!";

        if (abs($sum - $order->order_total) >= 0.005)
            return "Error: incorrect sum!";

        $order->add_history(JText::_('J2STORE_CALLBACK_RESPONSE_RECEIVED'));
        $order->update_status($this->params->get('payment_status', 4));

        //save the data
        $order->store();
        //clear cart
        $order->empty_cart();

        return "OK ".md5($id.$secret);
    }

    protected function getTaxRate($tax, $price, $is_shipping)
    {
        $tax = (float) $tax;
        $price = (float) $price;

        if ($is_shipping)
            return round(($tax/($price/100)), 0);
        else
            return ($this->order->is_including_tax)
                ? round(($tax*100/($price-$tax)), 0)
                : round(($tax/(($price - $tax)/100)), 0);
    }
}
