<?php

/**
 * Common PayKeeper payment system PHP class for using in PayKeeper payment modules for different CMS
 */
class PaykeeperPayment
{
    /**
     * @var array Fiscal cart items
     */
    public $fiscalCart = [];

    /**
     * @var float Total order amount
     */
    public $orderTotal = 0;

    /**
     * @var float Shipping price
     */
    public $shippingPrice = 0;

    /**
     * @var bool Flag indicating if taxes are used
     */
    public $useTaxes = false;

    /**
     * @var bool Flag indicating if delivery is used
     */
    public $useDelivery = false;

    /**
     * @var int Index of the delivery item in the fiscal cart
     */
    public $deliveryIndex = -1;

    /**
     * @var int Index of a single item in the fiscal cart
     */
    public $singleItemIndex = -1;

    /**
     * @var int Index of an item with more than one quantity in the fiscal cart
     */
    public $moreThanOneItemIndex = -1;

    /**
     * @var array|null Order parameters
     */
    public $orderParams = null;

    /**
     * @var array Discounts information
     */
    public $discounts = [];

    /**
     * @var string Language for the payment system
     */
    public $lang = 'ru';

    /**
     * Set order parameters.
     *
     * @param float $orderTotal Total order amount
     * @param string $clientId Client ID
     * @param string $orderId Order ID
     * @param string $clientEmail Client email
     * @param string $clientPhone Client phone
     * @param string $serviceName Service name
     * @param string $formUrl Form URL
     * @param string $secretKey Secret key
     */
    public function setOrderParams(
        $orderTotal = 0,
        $clientId = '',
        $orderId = '',
        $clientEmail = '',
        $clientPhone = '',
        $serviceName = '',
        $formUrl = '',
        $secretKey = ''
    ) {
        $this->setOrderTotal($orderTotal);
        $this->orderParams = [
            'sum' => $orderTotal,
            'clientid' => $clientId,
            'orderid' => $orderId,
            'service_name' => $serviceName,
            'client_email' => $clientEmail,
            'client_phone' => $clientPhone,
            'form_url' => $formUrl,
            'secret_key' => $secretKey,
        ];
    }

    /**
     * Get a specific order parameter.
     *
     * @param string $value Parameter key
     * @return mixed|false Parameter value or false if not found
     */
    public function getOrderParams($value)
    {
        return array_key_exists($value, $this->orderParams) ? $this->orderParams[$value] : false;
    }

    /**
     * Update fiscal cart.
     *
     * @param string $ftype Form type ('create' or 'order')
     * @param string $name Item name
     * @param float $price Item price
     * @param int $quantity Item quantity
     * @param float $sum Item sum
     * @param string $tax Tax rate
     */
    public function updateFiscalCart(
        $ftype,
        $name = '',
        $price = 0,
        $quantity = 0,
        $tax = 'none',
        $item_type = 'goods'
    ) {
        if ($ftype === 'create') {
            $name = str_replace(["\n ", "\r "], '', $name);
        }
        $priceFormat = number_format($price, 2, '.', '');
        $sumFormat = number_format($price * $quantity, 2, '.', '');
        $this->fiscalCart[] = [
            'name' => $name,
            'price' => $priceFormat,
            'quantity' => $quantity,
            'sum' => $sumFormat,
            'tax' => $tax,
            'item_type' => $item_type,
        ];
    }

    /**
     * Get the fiscal cart.
     *
     * @return array Fiscal cart items
     */
    public function getFiscalCart()
    {
        return $this->fiscalCart;
    }

    /**
     * Apply discounts to the fiscal cart.
     *
     * @param bool $discountEnabledFlag Flag indicating if discounts are enabled
     */
    public function setDiscounts($discountEnabledFlag)
    {
        $discountModifierValue = 1;
        $shippingIncluded = false;

        if ($discountEnabledFlag) {
            if ($this->getFiscalCartSum(false) > 0) {
                if ($this->getOrderTotal() >= $this->getShippingPrice()) {
                    if ($this->getFiscalCartSum(false) > 0) {
                        $discountModifierValue = ($this->getOrderTotal() - $this->getShippingPrice()) / $this->getFiscalCartSum(false);
                    }
                } else {
                    if ($this->getFiscalCartSum(true) > 0) {
                        $discountModifierValue = $this->getOrderTotal() / $this->getFiscalCartSum(true);
                        $shippingIncluded = true;
                    }
                }

                if ($discountModifierValue < 1) {
                    foreach ($this->fiscalCart as $pos => $item) {
                        if (!$shippingIncluded && $pos == $this->deliveryIndex) {
                            continue;
                        }
                        if ($item['quantity'] > 0) {
                            $price = $item['price'] * $discountModifierValue;
                            $this->fiscalCart[$pos]['price'] = number_format($price, 2, '.', '');
                            $sum = $this->fiscalCart[$pos]['price'] * $this->fiscalCart[$pos]['quantity'];
                            $this->fiscalCart[$pos]['sum'] = number_format($sum, 2, '.', '');
                        }
                    }
                }
            }
        }
    }

    /**
     * Correct the precision of the fiscal cart sums to match the total order amount.
     */
    public function correctPrecision()
    {
        $fiscalCartSum = $this->getFiscalCartSum(true);
        $totalSum = $this->getOrderTotal();
        $diffValue = $totalSum - $fiscalCartSum;

        if (abs($diffValue) >= 0.005) {
            $diffSum = number_format($diffValue, 2, '.', '');

            if ($this->useDelivery) {
                $this->correctPriceOfCartItem($diffSum, count($this->fiscalCart) - 1);
            } else {
                if ($this->singleItemIndex >= 0) {
                    $this->correctPriceOfCartItem($diffSum, $this->singleItemIndex);
                } elseif ($this->moreThanOneItemIndex >= 0) {
                    $this->splitCartItem($this->moreThanOneItemIndex);
                    $this->correctPriceOfCartItem($diffSum, count($this->fiscalCart) - 1);
                } else {
                    $modifyValue = $diffSum > 0 ? $totalSum / $fiscalCartSum : $fiscalCartSum / $totalSum;
                    foreach ($this->fiscalCart as $pos => $item) {
                        if ($item['quantity'] > 0) {
                            $sum = $item['sum'] * $modifyValue;
                            $this->fiscalCart[$pos]['sum'] *= number_format($sum, 2, '.', '');
                            $price = $this->fiscalCart[$pos]['sum'] / $item['quantity'];
                            $this->fiscalCart[$pos]['price'] = number_format($price, 2, '.', '');
                        }
                    }
                }
            }
        }
    }

    /**
     * Set the total order amount.
     *
     * @param float $value Total order amount
     */
    public function setOrderTotal($value)
    {
        $this->orderTotal = $value;
    }

    /**
     * Get the total order amount.
     *
     * @return float Total order amount
     */
    public function getOrderTotal()
    {
        return $this->orderTotal;
    }

    /**
     * Set the shipping price.
     *
     * @param float $value Shipping price
     */
    public function setShippingPrice($value)
    {
        $this->shippingPrice = $value;
    }

    /**
     * Get the shipping price.
     *
     * @return float Shipping price
     */
    public function getShippingPrice()
    {
        return $this->shippingPrice;
    }

    /**
     * Get the payment form type.
     *
     * @return string Payment form type ('create' or 'order')
     */
    public function getPaymentFormType()
    {
        return strpos($this->orderParams['form_url'], '/order/inline') === false ? 'create' : 'order';
    }

    /**
     * Enable the use of taxes.
     */
    public function setUseTaxes()
    {
        $this->useTaxes = true;
    }

    /**
     * Check if taxes are used.
     *
     * @return bool True if taxes are used, false otherwise
     */
    public function getUseTaxes()
    {
        return $this->useTaxes;
    }

    /**
     * Set the language for the payment system.
     *
     * @param string $lang Language code (e.g., 'ru')
     */
    public function setLang($lang)
    {
        $this->lang = $lang;
    }

    /**
     * Get the language for the payment system.
     *
     * @return string Language code
     */
    public function getLang()
    {
        return $this->lang;
    }

    /**
     * Correct the price of a specific cart item.
     *
     * @param float $value Value to correct
     * @param int $index Index of the cart item
     */
    private function correctPriceOfCartItem($value, $index)
    {
        $this->fiscalCart[$index]['sum'] = number_format($this->fiscalCart[$index]['sum'] + $value, 2, '.', '');
        $this->fiscalCart[$index]['price'] = number_format($this->fiscalCart[$index]['sum'] / $this->fiscalCart[$index]['quantity'], 2, '.', '');
    }

    /**
     * Get the sum of the fiscal cart.
     *
     * @param bool $shippingIncluded Flag indicating if shipping is included
     * @return float Sum of the fiscal cart
     */
    private function getFiscalCartSum($shippingIncluded = false)
    {
        $sum = 0;

        foreach ($this->fiscalCart as $pos => $item) {
            if (!$shippingIncluded && $pos == $this->deliveryIndex) {
                continue;
            }
            $sum += $item['sum'];
        }

        return number_format($sum, 2, '.', '');
    }

    /**
     * Split a cart item into individual items.
     *
     * @param int $index Index of the cart item to split
     */
    private function splitCartItem($index)
    {
        $clonedItem = $this->fiscalCart[$index];
        $clonedItem['quantity'] = $clonedItem['quantity'] - 1;
        $clonedItem['sum'] = number_format($clonedItem['price'] * $clonedItem['quantity'], 2, '.', '');
        $this->fiscalCart[$index]['quantity'] = 1;
        $this->fiscalCart[$index]['sum'] = $this->fiscalCart[$index]['price'];
        $this->fiscalCart[] = $clonedItem;
    }

    /**
     * Encode the fiscal cart as a JSON string.
     *
     * @return string JSON-encoded fiscal cart
     */
    public function getFiscalCartEncoded()
    {
        return json_encode($this->fiscalCart, JSON_UNESCAPED_UNICODE);
    }

    /**
     * Get the payment form parameters as a query string.
     *
     * @return string Payment form parameters
     */
    public function getPaymentFormParams()
    {
        $orderParams = $this->orderParams;
        $orderParams['cart'] = $this->getFiscalCartEncoded();
        return http_build_query($orderParams);
    }

    /**
     * Gets the default payment form.
     *
     * @param string $payment_form_sign
     * @return string
     */
    public function getDefaultPaymentForm($payment_form_sign)
    {
        if ($this->getPaymentFormType() === "create") {
            $form = '
                <h3>Сейчас Вы будете перенаправлены на страницу банка.</h3>
                <form name="paykeeper_form" id="paykeeper_form" action="' . $this->getOrderParams("form_url") . '" accept-charset="utf-8" method="post">
                    <input type="hidden" name="sum" value="' . $this->getOrderTotal() . '"/>
                    <input type="hidden" name="orderid" value="' . $this->getOrderParams("orderid") . '"/>
                    <input type="hidden" name="clientid" value="' . $this->getOrderParams("clientid") . '"/>
                    <input type="hidden" name="client_email" value="' . $this->getOrderParams("client_email") . '"/>
                    <input type="hidden" name="client_phone" value="' . $this->getOrderParams("client_phone") . '"/>
                    <input type="hidden" name="service_name" value="' . $this->getOrderParams("service_name") . '"/>
                    <input type="hidden" name="cart" value="' . htmlentities($this->getFiscalCartEncoded(), ENT_QUOTES) . '"/>
                    <input type="hidden" name="sign" value="' . $payment_form_sign . '"/>
                    <input type="submit" class="btn btn-default" value="Оплатить"/>
                </form>
                <script type="text/javascript">
                    function PayKeeperFormEvent(func) {
                        if (window.addEventListener) {
                            window.addEventListener("load", func, false);
                        } else if (window.attachEvent) {
                            window.attachEvent("onload", func);
                        } else {
                            var oldonload = window.onload;
                            window.onload = function() {
                                if (oldonload) {
                                    oldonload();
                                }
                                func();
                            };
                        }
                    }
            
                    function PayKeeperFormFunc() {
                        setTimeout(function() {
                            document.forms["paykeeper_form"].submit();
                        }, 2000);
                    }

                    PayKeeperFormEvent(PayKeeperFormFunc);
                </script>';
        } else {
            $payment_parameters = [
                "clientid" => $this->getOrderParams("clientid"),
                "orderid" => $this->getOrderParams('orderid'),
                "sum" => $this->getOrderTotal(),
                "client_phone" => $this->getOrderParams("phone"),
                "phone" => $this->getOrderParams("phone"),
                "client_email" => $this->getOrderParams("client_email"),
                "cart" => $this->getFiscalCartEncoded(),
            ];

            $query = http_build_query($payment_parameters);

            if (function_exists("curl_init")) {
                $ch = curl_init();
                curl_setopt($ch, CURLOPT_URL, $this->getOrderParams("form_url"));
                curl_setopt($ch, CURLOPT_POST, 1);
                curl_setopt($ch, CURLOPT_FAILONERROR, true);
                curl_setopt($ch, CURLOPT_POSTFIELDS, $query);
                curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
                curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, 0);
                $result = curl_exec($ch);
                if (curl_errno($ch)) {
                    $error_msg = curl_error($ch);
                }
                curl_close($ch);
                if (isset($error_msg)) {
                    $form = "INTERNAL ERROR: $error_msg";
                } else {
                    $form = $result;
                }
            } elseif (ini_get('allow_url_fopen')) {
                $query_options = [
                    'https' => [
                        'method'    => 'POST',
                        'header'    => 'Content-type: application/x-www-form-urlencoded',
                        'content'   => $query,
                    ]
                ];
                $context = stream_context_create($query_options);
                $form = file_get_contents($this->getOrderParams('form_url'), false, $context);
            } else {
                $form = 'INTERNAL ERROR: the curl php module is not installed or option allow_url_fopen is not set in php.ini';
            }
        }

        return empty($form) ? 'An error occurred while initializing the payment' : $form;
    }
}
