<?php
// This file is part of Moodle - https://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <https://www.gnu.org/licenses/>.

/**
 * Plugin administration pages are defined here.
 *
 * @package     paygw_paykeeper
 * @copyright   2024 PayKeeper Support <support@paykeeper.ru>
 * @license     https://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

use core_payment\helper;
use paygw_paykeeper\notifications;

require("../../../config.php");
global $CFG, $USER, $DB;

require_once($CFG->libdir . '/filelib.php');

defined('MOODLE_INTERNAL') || die();

if ($_SERVER['REQUEST_METHOD'] != 'POST') {
    die('ERROR. Invalid method request: ' . $_SERVER['REQUEST_METHOD']);
}

$requiredFields = ['id', 'sum', 'clientid', 'orderid', 'key'];
foreach ($requiredFields as $field) {
    if (empty($_POST[$field])) {
        die('ERROR. Invalid params in request: ' . $field);
    }
}

$invoiceid = required_param('id', PARAM_INT);
$outsumm = required_param('sum', PARAM_FLOAT);
$clientid = required_param('clientid', PARAM_TEXT);
$paymentid = required_param('orderid', PARAM_ALPHANUMEXT);
$key = required_param('key', PARAM_ALPHANUM);

if (isset($_POST['bank_id'])) {
    $bank_id = required_param(bank_id, PARAM_ALPHANUMEXT);
}

// Get paymentid.
if (!$paykeepertx = $DB->get_record('paygw_paykeeper', ['paymentid' => $paymentid])) {
    die('ERROR. Not a valid transaction id');
}

// Get payment data.
if (!$payment = $DB->get_record('payments', ['id' => $paykeepertx->paymentid])) {
    die('ERROR. Not a valid payment.');
}

$component   = $payment->component;
$paymentarea = $payment->paymentarea;
$itemid      = $payment->itemid;
$paymentid   = $payment->id;
$userid      = $payment->userid;

// Get config.
$config = (object) helper::get_gateway_configuration($component, $paymentarea, $itemid, 'paykeeper');
$payable = helper::get_payable($component, $paymentarea, $itemid);

$secret_seed = $config->pksecret;

if ($key != md5($invoiceid . number_format($outsumm, 2, ".", "")
        . $clientid . $paymentid . $secret_seed))
{
    echo "ERROR. Hash mismatch";
    exit;
}

$nextpay = null;
if ($config->recurrent == 1 && $config->recurrentperiod > 0 && isset($bank_id)) {
    $paykeepertx->recurrent = time() + $config->recurrentperiod;
    $nextpay = userdate($paykeepertx->recurrent, "%d %B %Y, %k:%M");
    $paykeepertx->bankid = $bank_id;
    $DB->update_record('paygw_paykeeper', $paykeepertx);
    unset($paykeepertx->recurrent);
    $reason = 'Success recurrent';
} else {
    $reason = 'Success completed';
}

if ($paykeepertx->invoiceid) {
    // Save new payment.
    $newpaymentid = helper::save_payment(
        $payable->get_account_id(),
        $component,
        $paymentarea,
        $itemid,
        $userid,
        $outsumm,
        $payment->currency,
        'paykeeper'
    );

    // Make new transaction.
    $paykeepertx->paymentid = $newpaymentid;
    $paykeepertx->timecreated = time();
    $paykeepertx->id = $DB->insert_record('paygw_paykeeper', $paykeepertx);
    $reason = 'Success completed';
} else {
    // Update payment.
    $payment->amount = $outsumm;
    $payment->timemodified = time();
    $DB->update_record('payments', $payment);
    $newpaymentid = $paymentid;
}

// Deliver order.
helper::deliver_order($component, $paymentarea, $itemid, $newpaymentid, $userid);

// Notify user.
notifications::notify(
    $userid,
    $outsumm,
    $payment->currency,
    $newpaymentid,
    $reason,
    $nextpay
);

// Write to DB.
$paykeepertx->success = 1;
$paykeepertx->invoiceid = $invoiceid;
$DB->update_record('paygw_paykeeper', $paykeepertx);

$hash = md5($invoiceid . $secret_seed);
die("OK $hash");
