<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Contains class for paykeeper payment gateway.
 *
 * @package    paygw_paykeeper
 * @copyright  2024 PayKeeper Support <support@paykeeper.ru>
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace paygw_paykeeper;

/**
 * The gateway class for paykeeper payment gateway.
 *
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
class gateway extends \core_payment\gateway {
    /**
     * Configuration form for currency
     */
    public static function get_supported_currencies(): array {
        // 3-character ISO-4217: https://en.wikipedia.org/wiki/ISO_4217#Active_codes.
        return [
            'RUB',
        ];
    }

    /**
     * Configuration form for the gateway instance
     *
     * Use $form->get_mform() to access the \MoodleQuickForm instance
     *
     * @param \core_payment\form\account_gateway $form
     */
    public static function add_configuration_to_gateway_form(\core_payment\form\account_gateway $form): void {
        $mform = $form->get_mform();

        $mform->addElement('text', 'pkserver', get_string('pkserver', 'paygw_paykeeper'), ['size' => 50]);
        $mform->setType('pkserver', PARAM_TEXT);
        $mform->addRule('pkserver', get_string('required'), 'required', null, 'client');

        $mform->addElement('text', 'pksecret', get_string('pksecret', 'paygw_paykeeper'), ['size' => 50]);
        $mform->setType('pksecret', PARAM_TEXT);
        $mform->addRule('pksecret', get_string('required'), 'required', null, 'client');

        $mform->addElement('text', 'pklogin', get_string('pklogin', 'paygw_paykeeper'), ['size' => 50]);
        $mform->setType('pklogin', PARAM_TEXT);
        $mform->addRule('pklogin', get_string('required'), 'required', null, 'client');

        $mform->addElement('text', 'pkpassword', get_string('pkpassword', 'paygw_paykeeper'), ['size' => 50]);
        $mform->setType('pkpassword', PARAM_TEXT);
        $mform->addRule('pkpassword', get_string('required'), 'required', null, 'client');

        $options = [
            'none' => get_string('no'),
            'vat0' => "0%",
            'vat5' => "5%",
            'vat7' => "7%",
            'vat10' => "10%",
            'vat20' => "20%",
            'vat105' => "5/105",
            'vat107' => "7/107",
            'vat110' => "10/110",
            'vat120' => "20/120"
        ];
        $mform->addElement(
            'select',
            'vatcode',
            get_string('vatcode', 'paygw_paykeeper'),
            $options
        );
        $mform->setType('vatcode', PARAM_TEXT);
        $mform->addHelpButton('vatcode', 'vatcode', 'paygw_paykeeper');

        $mform->addElement(
            'advcheckbox',
            'recurrent',
            get_string('recurrent', 'paygw_paykeeper')
        );
        $mform->setType('recurrent', PARAM_INT);
        $mform->addHelpButton('recurrent', 'recurrent', 'paygw_paykeeper');

        $options = [0 => get_string('no')];
        for ($i = 1; $i <= 28; $i++) {
            $options[] = $i;
        }
        $mform->addElement(
            'select',
            'recurrentday',
            get_string('recurrentday', 'paygw_paykeeper'),
            $options,
        );
        $mform->addHelpButton('recurrentday', 'recurrentday', 'paygw_paykeeper');
        $mform->setDefault('recurrentday', 1);
        $mform->hideIf('recurrentday', 'recurrent', "neq", 1);

        $mform->addElement('duration', 'recurrentperiod', get_string('recurrentperiod', 'paygw_paykeeper'), ['optional' => false]);
        $mform->setType('recurrentperiod', PARAM_INT);
        $mform->hideIf('recurrentperiod', 'recurrent', "neq", 1);
        $mform->hideIf('recurrentperiod', 'recurrentday', "neq", 0);
        $mform->addHelpButton('recurrentperiod', 'recurrentperiod', 'paygw_paykeeper');

        $options = [
            'last' => get_string('recurrentcost1', 'paygw_paykeeper'),
            'fee' => get_string('recurrentcost2', 'paygw_paykeeper'),
            'suggest' => get_string('recurrentcost3', 'paygw_paykeeper'),
        ];
        $mform->addElement(
            'select',
            'recurrentcost',
            get_string('recurrentcost', 'paygw_paykeeper'),
            $options,
        );
        $mform->setType('recurrentcost', PARAM_TEXT);
        $mform->addHelpButton('recurrentcost', 'recurrentcost', 'paygw_paykeeper');
        $mform->setDefault('recurrentcost', 'fee');
        $mform->hideIf('recurrentcost', 'recurrent', "neq", 1);

        $plugininfo = \core_plugin_manager::instance()->get_plugin_info('report_payments');
        if (!isset($plugininfo->versiondisk) || $plugininfo->versiondisk < 3024070800) {
            $mform->addElement('static', 'noreport', null, get_string('noreportplugin', 'paygw_paykeeper'));
        }

        $mform->addElement(
            'advcheckbox',
            'sendlinkmsg',
            get_string('sendlinkmsg', 'paygw_paykeeper')
        );
        $mform->setType('sendlinkmsg', PARAM_INT);
        $mform->addHelpButton('sendlinkmsg', 'sendlinkmsg', 'paygw_paykeeper');
        $mform->setDefault('sendlinkmsg', 1);

        $mform->addElement('text', 'fixdesc', get_string('fixdesc', 'paygw_paykeeper'), ['size' => 50]);
        $mform->setType('fixdesc', PARAM_TEXT);
        $mform->addRule('fixdesc', get_string('maximumchars', '', 255), 'maxlength', 255, 'client');
        $mform->addHelpButton('fixdesc', 'fixdesc', 'paygw_paykeeper');

        $mform->addElement('static');

        $mform->addElement(
            'advcheckbox',
            'skipmode',
            get_string('skipmode', 'paygw_paykeeper')
        );
        $mform->setType('skipmode', PARAM_INT);
        $mform->addHelpButton('skipmode', 'skipmode', 'paygw_paykeeper');

        $mform->addElement(
            'advcheckbox',
            'passwordmode',
            get_string('passwordmode', 'paygw_paykeeper')
        );
        $mform->setType('passwordmode', PARAM_INT);
        $mform->disabledIf('passwordmode', 'skipmode', "neq", 0);

        $mform->addElement('passwordunmask', 'password', get_string('password', 'paygw_paykeeper'), ['size' => 20]);
        $mform->setType('password', PARAM_TEXT);
        $mform->addHelpButton('password', 'password', 'paygw_paykeeper');

        $mform->addElement(
            'advcheckbox',
            'usedetails',
            get_string('usedetails', 'paygw_paykeeper')
        );
        $mform->setType('usedetails', PARAM_INT);
        $mform->addHelpButton('usedetails', 'usedetails', 'paygw_paykeeper');

        $mform->addElement(
            'advcheckbox',
            'showduration',
            get_string('showduration', 'paygw_paykeeper')
        );
        $mform->setType('showduration', PARAM_INT);

        $mform->addElement(
            'advcheckbox',
            'fixcost',
            get_string('fixcost', 'paygw_paykeeper')
        );
        $mform->setType('fixcost', PARAM_INT);
        $mform->addHelpButton('fixcost', 'fixcost', 'paygw_paykeeper');

        $mform->addElement('text', 'suggest', get_string('suggest', 'paygw_paykeeper'), ['size' => 10]);
        $mform->setType('suggest', PARAM_TEXT);
        $mform->disabledIf('suggest', 'fixcost', "neq", 0);

        $mform->addElement('text', 'maxcost', get_string('maxcost', 'paygw_paykeeper'), ['size' => 10]);
        $mform->setType('maxcost', PARAM_TEXT);
        $mform->disabledIf('maxcost', 'fixcost', "neq", 0);

        global $CFG;
        $mform->addElement('html', '<div class="label-callback" style="background: pink; padding: 15px;">' .
            get_string('callback_url', 'paygw_paykeeper') . '<br>');
        $mform->addElement('html', $CFG->wwwroot . '/payment/gateway/paykeeper/callback.php<br>');
        $mform->addElement('html', get_string('callback_help', 'paygw_paykeeper') . '</div><br>');

        $plugininfo = \core_plugin_manager::instance()->get_plugin_info('paygw_paykeeper');
        $donate = get_string('donate', 'paygw_paykeeper', $plugininfo);
        $mform->addElement('html', $donate);
    }

    /**
     * Validates the gateway configuration form.
     *
     * @param \core_payment\form\account_gateway $form
     * @param \stdClass $data
     * @param array $files
     * @param array $errors form errors (passed by reference)
     */
    public static function validate_gateway_form(
        \core_payment\form\account_gateway $form,
        \stdClass $data,
        array $files,
        array &$errors
    ): void {
        if ($data->enabled && (empty($data->pkserver) || empty($data->pksecret) || empty($data->pklogin)
                || empty($data->pkpassword)))
        {
            $errors['enabled'] = get_string('gatewaycannotbeenabled', 'payment');
        }
        if ($data->maxcost && $data->maxcost < $data->suggest) {
            $errors['maxcost'] = get_string('maxcosterror', 'paygw_paykeeper');
        }
        if (!$data->suggest && $data->recurrentcost == 'suggest' && $data->recurrent) {
            $errors['suggest'] = get_string('suggesterror', 'paygw_paykeeper');
        }
        if (!$data->recurrentperiod && $data->recurrent && !$data->recurrentday) {
            $errors['recurrentperiod'] = get_string('recurrentperioderror', 'paygw_paykeeper');
        }
    }
}
