<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * recurrent payments
 *
 * @package    paygw_paykeeper
 * @copyright  2024 PayKeeper Support <support@paykeeper.ru>
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace paygw_paykeeper\task;

defined('MOODLE_INTERNAL') || die();

use core_payment\helper;
use paygw_paykeeper\notifications;

/**
 * @var $CFG
 */
require_once($CFG->libdir . '/filelib.php');

/**
 * Default tasks.
 *
 * @package    paygw_paykeeper
 * @copyright  2024 PayKeeper Support <support@paykeeper.ru>
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
class recurrent_payments extends \core\task\scheduled_task {
    /**
     * Get a descriptive name for this task (shown to admins).
     *
     * @return string
     */
    public function get_name() {
        return get_string('pluginname', 'paygw_paykeeper');
    }
    /**
     * Execute.
     */
    public function execute() {
        global $DB, $CFG;

        mtrace('Start');

        // Stage One.
        $stime = strtotime(date('d-M-Y H:00', strtotime("+1day")));
        $ctime = strtotime(date('d-M-Y H:00', strtotime("+1day1hour")));

        $paykeepertx = $DB->get_records_sql('SELECT * FROM {paygw_paykeeper} WHERE (success=1 OR success=3) ' .
            'AND recurrent>=? AND recurrent<?', [ $stime, $ctime ]);

        foreach ($paykeepertx as $data) {
            // Get payment data.
            if (!$payment = $DB->get_record('payments', ['id' => $data->paymentid])) {
                mtrace("$data->paymentid not found");
                continue;
            }

            $component   = $payment->component;
            $paymentarea = $payment->paymentarea;
            $itemid      = $payment->itemid;
            $userid      = $payment->userid;

            // Get config.
            $config = (object) helper::get_gateway_configuration($component, $paymentarea, $itemid, 'paykeeper');
            $payable = helper::get_payable($component, $paymentarea, $itemid);
            $surcharge = helper::get_gateway_surcharge('paykeeper');// In case user uses surcharge.

            switch ($config->recurrentcost) {
                case 'suggest':
                    $cost = $config->suggest;
                    break;
                case 'last':
                    $cost = $payment->amount;
                    break;
                default:
                    $cost = helper::get_rounded_cost($payable->get_amount(), $payable->get_currency(), $surcharge);
            }

            // Notify user.
            notifications::notify(
                $userid,
                $cost,
                $payment->currency,
                $data->paymentid,
                'Recurrent notify',
                userdate($data->recurrent, "%d %B %k:00")
            );

            mtrace("$data->paymentid notified");
        }

        // Stage Two.
        $ctime = strtotime(date('d-M-Y H:00', strtotime("+1hour")));

        $paykeepertx = $DB->get_records_sql('SELECT * FROM {paygw_paykeeper} WHERE (success=1 OR success=3) ' .
            'AND recurrent>0 AND recurrent < ?', [ $ctime ]);

        foreach ($paykeepertx as $data) {
            // To avoid abuse.
            sleep(1);

            // Get payment data.
            if (!$payment = $DB->get_record('payments', ['id' => $data->paymentid])) {
                mtrace("$data->paymentid not found");
                continue;
            }

            $component   = $payment->component;
            $paymentarea = $payment->paymentarea;
            $itemid      = $payment->itemid;
            $paymentid   = $payment->id;
            $userid      = $payment->userid;

            // Get config.
            $config = (object) helper::get_gateway_configuration($component, $paymentarea, $itemid, 'paykeeper');
            $payable = helper::get_payable($component, $paymentarea, $itemid);
            $surcharge = helper::get_gateway_surcharge('paykeeper');// In case user uses surcharge.

            if (date('d') != $config->recurrentday && $config->recurrentday > 0) {
                mtrace("$data->paymentid too early");
                continue;
            }

            switch ($config->recurrentcost) {
                case 'suggest':
                    $cost = $config->suggest;
                    break;
                case 'last':
                    $cost = $payment->amount;
                    break;
                default:
                    $cost = helper::get_rounded_cost($payable->get_amount(), $payable->get_currency(), $surcharge);
            }

            $user = \core_user::get_user($userid);

            // Get token
            $location = 'https://' . $config->pkserver .  '/info/settings/token/';
            $options = [
                'CURLOPT_RETURNTRANSFER' => true,
                'CURLOPT_TIMEOUT' => 30,
                'CURLOPT_HTTP_VERSION' => CURL_HTTP_VERSION_1_1,
                'CURLOPT_SSLVERSION' => CURL_SSLVERSION_TLSv1_2,
                'CURLOPT_HTTPHEADER' => [
                    'Content-Type: application/x-www-form-urlencoded',
                ],
                'CURLOPT_HTTPAUTH' => CURLAUTH_BASIC,
                'CURLOPT_USERPWD' => $config->pklogin . ':' . $config->pkpassword,
            ];
            $curl = new \curl();
            $jsonresponse = $curl->get($location, $options);
            $response = json_decode($jsonresponse);
            if (!empty($response->token)) {
                // Make Order
                $sum = number_format($cost, 2, '.', '');

                $clientid = trim($user->firstname . ' ' . $user->lastname) . ' ID ' . $userid;
                $orderid = $paymentid;
                $service_name = '';
                $client_email = $user->email;
                $client_phone = $user->phone1 ? $user->phone1 : $user->phone2;
                $secret_key = $config->pksecret;
                $sign = hash('sha256' , $sum . $clientid . $orderid . $service_name . $client_email
                    . $client_phone . $secret_key);

                $post_params = [
                    'sum' => $sum,
                    'clientid' => $clientid,
                    'orderid' => $orderid,
                    'client_email' => $client_email,
                    'client_phone' => $client_phone,
                    'phone' => $client_phone,
                    'service_name' => $service_name,
                    'cart' => json_encode([
                        'name' => "Recurrent payment " . $data->paymentid,
                        'price' => $sum,
                        'quantity' => 1,
                        'sum' => $sum,
                        'tax' => $config->vatcode
                    ]),
                    'sign' => $sign,
                    'bank_id' => $data->bankid,
                    'token' => $response->token
                ];
            }

            if (isset($post_params)) {
                // Make payment.
                $location = 'https://' . $config->pkserver .  '/change/binding/execute/';
                $options = [
                    'CURLOPT_RETURNTRANSFER' => true,
                    'CURLOPT_TIMEOUT' => 30,
                    'CURLOPT_HTTP_VERSION' => CURL_HTTP_VERSION_1_1,
                    'CURLOPT_SSLVERSION' => CURL_SSLVERSION_TLSv1_2,
                    'CURLOPT_HTTPHEADER' => [
                        'Content-Type: application/x-www-form-urlencoded',
                    ],
                    'CURLOPT_HTTPAUTH' => CURLAUTH_BASIC,
                    'CURLOPT_USERPWD' => $config->pklogin . ':' . $config->pkpassword,
                ];
                $curl = new \curl();
                $jsondata = json_encode($post_params);
                $jsonresponse = $curl->post($location, $jsondata, $options);

                $response = json_decode($jsonresponse);

                if ($response->result == 'success') {
                    mtrace("$data->paymentid done.");
                    // Notify user.
                    notifications::notify(
                        $userid,
                        $cost,
                        $payment->currency,
                        $data->paymentid,
                        'Recurrent created'
                    );
                }
            }

            if (isset($response) && (empty($response->token) || $response->result == 'fail')) {
                echo serialize($jsonresponse) . "\n";
                mtrace("$data->paymentid error");
                $data->recurrent = 0;
                $DB->update_record('paygw_paykeeper', $data);
                // Notify user.
                notifications::notify(
                    $userid,
                    $cost,
                    $payment->currency,
                    $data->paymentid,
                    'Recurrent error'
                );
            }
        }

        mtrace('End');
    }
}
