<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Redirects user to the payment page
 *
 * @package   paygw_paykeeper
 * @copyright 2024 PayKeeper Support <support@paykeeper.ru>
 * @license   http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

use core_payment\helper;
use paygw_paykeeper\notifications;
use paygw_paykeeper\lib\paykeeper_class;

require_once(__DIR__ . '/../../../config.php');
global $CFG, $USER, $DB;
require_once($CFG->libdir . '/filelib.php');

require_login();
require_sesskey();

/**
 * @var $PAGE
 * @var $SCRIPT
 */

$pk_obj = new paykeeper_class();

$userid = $USER->id;

$component   = required_param('component', PARAM_COMPONENT);
$paymentarea = required_param('paymentarea', PARAM_AREA);
$itemid      = required_param('itemid', PARAM_INT);
$description = required_param('description', PARAM_TEXT);

$password    = optional_param('password', null, PARAM_TEXT);
$skipmode    = optional_param('skipmode', 0, PARAM_INT);
$costself    = optional_param('costself', null, PARAM_TEXT);

$config = (object) helper::get_gateway_configuration($component, $paymentarea, $itemid, 'paykeeper');
$payable = helper::get_payable($component, $paymentarea, $itemid);// Get currency and payment amount.
$currency = $payable->get_currency();
$surcharge = helper::get_gateway_surcharge('paykeeper');// In case user uses surcharge.
$cost = helper::get_rounded_cost($payable->get_amount(), $payable->get_currency(), $surcharge);

// Check self cost if not fixcost.
if (!empty($costself) && !$config->fixcost) {
    $cost = $costself;
}

// Check maxcost.
if ($config->maxcost && $cost > $config->maxcost) {
    $cost = $config->maxcost;
}

$cost = number_format($cost, 2, '.', '');

// Get course and groups for user.
if ($component == "enrol_fee") {
    $cs = $DB->get_record('enrol', ['id' => $itemid]);
    $cs->course = $cs->courseid;
} else if ($component == "mod_gwpayments") {
    $cs = $DB->get_record('gwpayments', ['id' => $itemid]);
} else if ($paymentarea == "cmfee") {
    $cs = $DB->get_record('course_modules', ['id' => $itemid]);
} else if ($paymentarea == "sectionfee") {
    $cs = $DB->get_record('course_sections', ['id' => $itemid]);
}
$groupnames = '';
if (!empty($cs->course)) {
    $courseid = $cs->course;
    if ($gs = groups_get_user_groups($courseid, $userid, true)) {
        foreach ($gs as $gr) {
            foreach ($gr as $g) {
                $groups[] = groups_get_group_name($g);
            }
        }
        if (isset($groups)) {
            $groupnames = implode(',', $groups);
        }
    }
} else {
    $courseid = '';
}

// Write tx to DB.
$paykeepertx = new stdClass();
$paykeepertx->courseid = $courseid;
$paykeepertx->groupnames = $groupnames;
$paykeepertx->timecreated = time();
if (!$transactionid = $DB->insert_record('paygw_paykeeper', $paykeepertx)) {
    throw new \moodle_exception(get_string('error_txdatabase', 'paygw_paykeeper'), 'paygw_paykeeper');
}
$paykeepertx->id = $transactionid;

// Set the context of the page.
$PAGE->set_url($SCRIPT);
$PAGE->set_context(context_system::instance());

// Check passwordmode or skipmode.
if (!empty($password) || $skipmode) {
    $success = false;
    if ($config->skipmode) {
        $success = true;
    } else if (!empty($cs->password)) {
        // Check module password.
        if ($password === $cs->password) {
            $success = true;
        }
    } else if ($config->passwordmode && !empty($config->password)) {
        // Check payment password.
        if ($password === $config->password) {
            $success = true;
        }
    }

    // Build redirect.
    $url = helper::get_success_url($component, $paymentarea, $itemid);

    if ($success) {
        // Make fake pay.
        $paymentid = helper::save_payment(
            $payable->get_account_id(),
            $component,
            $paymentarea,
            $itemid,
            $userid,
            0,
            $payable->get_currency(),
            'paykeeper'
        );
        helper::deliver_order($component, $paymentarea, $itemid, $paymentid, $userid);

        // Write to DB.
        $paykeepertx->success = 2;
        $paykeepertx->paymentid = $paymentid;
        $DB->update_record('paygw_paykeeper', $paykeepertx);

        redirect($url, get_string('password_success', 'paygw_paykeeper'), 0, 'success');
    } else {
        redirect($url, get_string('password_error', 'paygw_paykeeper'), 0, 'error');
    }
    exit;
}

// Save payment.
$paymentid = helper::save_payment(
    $payable->get_account_id(),
    $component,
    $paymentarea,
    $itemid,
    $userid,
    $cost,
    $payable->get_currency(),
    'paykeeper'
);

$paykeepertx->paymentid = $paymentid;
$DB->update_record('paygw_paykeeper', $paykeepertx);

$pk_obj->setOrderParams(
    number_format($cost, 2, '.', ''), // order_total
    trim($USER->firstname . ' ' . $USER->lastname), // clientid
    $paymentid, // orderid
    $USER->email, // client_email
    $USER->phone1 ? $USER->phone1 : $USER->phone2, // client_phone
    '', // service_name
    'https://' . $config->pkserver . '/create', // form_url
    $config->pksecret // secret_key
);

$pk_obj->order_params['user_result_callback'] = $CFG->wwwroot . '/payment/gateway/paykeeper/return.php?id=' . $paymentid;

if ($config->recurrent == 1 && $config->recurrentperiod > 0) {
    $pk_obj->order_params['msgtype'] = 'createbinding';
    $pk_obj->order_params['clientid'] = $pk_obj->getOrderParams("clientid") . ' ID ' . $userid;
}

$pk_obj->updateFiscalCart(
    $pk_obj->getPaymentFormType(),
    $description,
    $cost,
    1,
    0,
    $config->vatcode
);

//echo '<pre>';
//echo var_export($payment, true);
//echo '</pre>';
//echo $pk_obj->getDefaultPaymentForm('');

$payment_form_sign = $pk_obj->getOrderTotal().
    $pk_obj->getOrderParams('clientid')     .
    $pk_obj->getOrderParams('orderid')      .
    $pk_obj->getOrderParams('service_name') .
    $pk_obj->getOrderParams('client_email') .
    $pk_obj->getOrderParams('client_phone') .
    $pk_obj->getOrderParams('secret_key');
$sign = hash('sha256' , $payment_form_sign);

$form = '';
if ($pk_obj->getPaymentFormType() == "create") { //create form
    $form = '
        <h3>Сейчас Вы будете перенаправлены на страницу банка.</h3>
        <form name="pay_form" id="pay_form" action="'.$pk_obj->getOrderParams("form_url").'" accept-charset="utf-8" method="post">
            <input type="hidden" name="sum" value = "'.$pk_obj->getOrderTotal().'"/>
            <input type="hidden" name="orderid" value = "'.$pk_obj->getOrderParams("orderid").'"/>
            <input type="hidden" name="clientid" value = "'.$pk_obj->getOrderParams("clientid").'"/>
            <input type="hidden" name="client_email" value = "'.$pk_obj->getOrderParams("client_email").'"/>
            <input type="hidden" name="client_phone" value = "'.$pk_obj->getOrderParams("client_phone").'"/>
            <input type="hidden" name="service_name" value = "'.$pk_obj->getOrderParams("service_name").'"/>
            <input type="hidden" name="cart" value = \''.htmlentities($pk_obj->getFiscalCartEncoded(),ENT_QUOTES).'\' />
            <input type="hidden" name="sign" value = "'.hash('sha256' , $payment_form_sign).'"/>
            <input type="hidden" name="user_result_callback" value = "'.$pk_obj->getOrderParams("user_result_callback").'"/>
            ';
    if ($pk_obj->getOrderParams("msgtype")) {
        $form .= '<input type="hidden" name="msgtype" value = "'.$pk_obj->getOrderParams("msgtype").'"/>';
    }
    $form .= '
            <input type="hidden" name="lang" value = "'.$pk_obj->getCurrentLang().'"/>
            <input type="submit" class="btn btn-default" value="Оплатить"/>
        </form>
        <script type="text/javascript">
            document.forms["pay_form"].submit();
        </script>';
}
else { //order form
    $payment_parameters = array(
        "clientid"=>$pk_obj->getOrderParams("clientid"),
        "orderid"=>$pk_obj->getOrderParams('orderid'),
        "sum"=>$pk_obj->getOrderTotal(),
        "client_phone"=>$pk_obj->getOrderParams("phone"),
        "phone"=>$pk_obj->getOrderParams("phone"),
        "client_email"=>$pk_obj->getOrderParams("client_email"),
        "cart"=>$pk_obj->getFiscalCartEncoded());
    $query = http_build_query($payment_parameters);
    if( function_exists( "curl_init" )) { //using curl
        $CR = curl_init();
        curl_setopt($CR, CURLOPT_URL, $pk_obj->getOrderParams("form_url"));
        curl_setopt($CR, CURLOPT_POST, 1);
        curl_setopt($CR, CURLOPT_FAILONERROR, true);
        curl_setopt($CR, CURLOPT_POSTFIELDS, $query);
        curl_setopt($CR, CURLOPT_RETURNTRANSFER, 1);
        curl_setopt($CR, CURLOPT_SSL_VERIFYPEER, 0);
        $result = curl_exec( $CR );
        $error = curl_error( $CR );
        if( !empty( $error )) {
            $form = "<br/><span class=message>"."INTERNAL ERROR:".$error."</span>";
        }
        else {
            $form = $result;
        }
        curl_close($CR);
    }
    else { //using file_get_contents
        if (!ini_get('allow_url_fopen')) {
            $form = "<br/><span class=message>"."INTERNAL ERROR: Option allow_url_fopen is not set in php.ini"."</span>";
        }
        else {
            $query_options = array(
                "https"=>array(
                    "method"=>"POST",
                    "header"=>"Content-type: application/x-www-form-urlencoded",
                    "content"=>$query
                )
            );
            $context = stream_context_create($query_options);
            $form = file_get_contents($pk_obj->getOrderParams("form_url"), false, $context);
        }
    }
}
if ($form  == "") {
    $form = '<h3>Произошла ошибка при инциализации платежа</h3>';
}

echo $form;
