<?php

/**
 * @package     EasyStore.Site
 * @subpackage  EasyStore.Sample
 *
 * @copyright   Copyright (C) 2023 JoomShaper <https://www.joomshaper.com>. All rights reserved.
 * @license     GNU General Public License version 3; see LICENSE
 */

namespace JoomShaper\Plugin\EasyStore\PayKeeper\Extension;


use Joomla\CMS\Log\Log;
use Joomla\Event\Event;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Application\CMSApplication;
use JoomShaper\Plugin\EasyStore\PayKeeper\Utils\PayKeeperConstants;
use JoomShaper\Plugin\EasyStore\PayKeeper\Helper\PayKeeperHelper;
use JoomShaper\Component\EasyStore\Administrator\Plugin\PaymentGatewayPlugin;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

class PayKeeperPayment extends PaymentGatewayPlugin
{
    /**
     * Verifies whether all the essential fields are configured before initiating the payment process, and  sets a boolean value,in the event arguments using the key `result`.
     *
     * @return void -- Returns true if all the necessary field is not empty, otherwise, returns false.
     *
     * @since 1.0.0
     */

    public function onBeforePayment(Event $event)
    {
        $constants              = new PayKeeperConstants();
        $pk_url                 = $constants->getUrl();
        $pk_secret              = $constants->getSecret();
        $isRequiredFieldsFilled = !empty($pk_secret) && !empty($pk_url);

        $event->setArgument('result', $isRequiredFieldsFilled);
    }

    /**
     * Initiate an event that will lead to a redirection to the checkout page.
     *
     * @param Event $event -- The event object that contains cart data required for payment processing.
     *
     * @since 1.0.0
     */

    public function onPayment(Event $event)
    {
        // Get the necessary data from `SampleConstants` which are needed to initiate payment process.
        $constants =        new PayKeeperConstants();
        $pk_url =           $constants->getUrl();
        $pk_secret =        $constants->getSecret();
        $pk_login =         $constants->getLogin();
        $pk_pass =          $constants->getPass();
        $pk_goods_vat =     $constants->getGoodsVat();
        $pk_service_vat =   $constants->getServiceVat();
        $pk_use_discount =  $constants->useDiscount();

        /**
         * Extracting payment-related data from the event arguments.
         * If the subject is not set in the arguments, a default empty object is used.
         */
        $arguments   = $event->getArguments();
        $paymentData = $arguments['subject'] ?: new \stdClass();

        // Array representing the cart items.
        $products             = $paymentData->items;
        // The overall transaction cost, including tax, shipping charge, and coupon discounts.
        $totalPrice           = $paymentData->total_price;
        // Easystore Order ID
        $orderID              = $paymentData->order_id;
        // Name of the store which is set in the Store Settings. Default value is : `EasyStore`
        $storeName            = $paymentData->store_name;
        // Tax Amount for the order.
        $tax                  = $paymentData->tax;
        // Currency which is set in the Store Settings.
        $currency             = $paymentData->currency;
        // Country ISO alpha2/alpha3 value
        $country              = $paymentData->country;
        // Shipping charge for the order.
        $shippingCharge       = $paymentData->shipping_charge;
        // Coupon Discount Amount for the order.
        $couponDiscountAmount = $paymentData->coupon_discount_amount;
        // Url of `EasyStore` Checkout Page
        $backToCheckoutPage   = $paymentData->back_to_checkout_page;
        // Decimal Separator which is set in the Store Settings. Default Value is '.'
        $decimalSeparator     = $paymentData->decimal_separator;
        // Thousand Separator which is set in the Store Settings. Default Value is ','
        $thousandsSeparator   = $paymentData->thousands_separator;
        // User Name
        $userName             = $paymentData->user_name;
        // User Phone Number
        $userNumber           = $paymentData->user_number;
        // User Email Address
        $userEmail            = $paymentData->user_email;

        /**
         * Define the URLs used for payment processing and notifications.
         */

        $redirectUrlAfterPayment = $constants->getSuccessUrl();  // User redirection after successfully completing the payment procedure on the payment portal.
        $cancelUrl               = $constants->getCancelUrl($orderID);  // User redirection upon payment cancellation
        $webhookUrl              = $constants->getWebHookUrl(); // URL for payment portal notifications

        try {

            /**
             *  In this try block, perform necessary tasks to create a checkout URL using the above variables for your payment, and then redirect the user.
             *
             *  📝 Ensure to transmit your $orderID as `metadata` or through any other supported process by your payment system while creating `$checkoutUrl`. This allows you to retrieve it in the onPaymentNotify() method for updating your payment status.
             */

            $pk_helper = new PayKeeperHelper();
            $pk_helper->setOrderTotal($paymentData->total_price_in_smallest_unit / 100);
            $pk_helper->setOrderParams(
                $pk_helper->getOrderTotal(),
                $userName,
                $orderID,
                $userEmail,
                $userNumber,
                '',
                $pk_url,
                $pk_secret
            );

            $vat = $pk_goods_vat;
            $tax_rate = 0;
            if ($paymentData->tax_in_smallest_unit) {
                $pk_helper->setUseTaxes();
                $tax_rate = $paymentData->tax_in_smallest_unit / $paymentData->subtotal;
                $taxes = $pk_helper->setTaxes($tax_rate, false);
                $vat = $taxes['tax'];
            }

            $index_cart = 0;
            foreach ($products as $product) {
                $price = 0;

                // Если есть цена со скидкой
                if (isset($product->discounted_price) && isset($product->discounted_price_in_smallest_unit)) {
                    $price = $product->discounted_price_in_smallest_unit / 100;
                }
                // Если есть обработанная цена
                elseif (isset($product->regular_price)) {
                    $price = $product->regular_price;
                }
                // Если есть обычная цена
                elseif (isset($product->price)) {
                    $price = $product->price;
                }
                // Если ни одна цена не задана, можно выбросить исключение или логировать ошибку
                else {
                    throw new Exception("Цена товара не указана");
                }

                if ($pk_helper->getUseTaxes()) {
                    $price += $price * $tax_rate / 100;
                }

                if ($product->quantity == 1 && $pk_helper->single_item_index < 0) {
                    $pk_helper->single_item_index = $index_cart;
                }
                if ($product->quantity > 1 && $pk_helper->more_then_one_item_index < 0) {
                    $pk_helper->more_then_one_item_index = $index_cart;
                }

                $pk_helper->updateFiscalCart(
                    $pk_helper->getPaymentFormType(),
                    $product->title,
                    $price,
                    $product->quantity,
                    0,
                    $vat
                );
                $index_cart++;
            }

            if ($shippingCharge) {
                if ($shippingCharge == $couponDiscountAmount) {
                    $couponDiscountAmount = 0;
                } else {
                    $pk_helper->setUseDelivery();
                    $pk_helper->updateFiscalCart(
                        $pk_helper->getPaymentFormType(),
                        $paymentData->shipping_method,
                        $shippingCharge,
                        1,
                        0,
                        $pk_service_vat
                    );
                    $pk_helper->delivery_index = $index_cart;
                    $pk_helper->fiscal_cart[$pk_helper->delivery_index]['item_type'] = 'service';
                }
            }

            if ($couponDiscountAmount || $pk_use_discount) {
                $pk_helper->setDiscounts(true);
            }

            $pk_helper->correctPrecision();

            $to_hash = number_format($pk_helper->getOrderTotal(), 2, '.', '').
                $pk_helper->getOrderParams('clientid')     .
                $pk_helper->getOrderParams('orderid')      .
                $pk_helper->getOrderParams('service_name') .
                $pk_helper->getOrderParams('client_email') .
                $pk_helper->getOrderParams('client_phone') .
                $pk_helper->getOrderParams('secret_key');
            $payment_form_sign = hash ('sha256' , $to_hash);
            // Default $this->app->redirect($checkoutUrl);
            echo $pk_helper->getDefaultPaymentForm($payment_form_sign);

        } catch (\Throwable $error) {

            // Log the error message with the Log class.
            Log::add($error->getMessage(), Log::ERROR, 'paykeeper.easystore');

            // Enqueue an error message to be displayed to the user.
            $this->app->enqueueMessage(Text::_('COM_EASYSTORE_PAYKEEPER_ERROR') . ' ' . $error->getMessage(), 'error');

            // Redirect the user to the checkout page to address the error.
            $this->app->redirect($paymentData->back_to_checkout_page);
        }
    }

    /**
     * Handles notifications received from the webhook / payment portal.
     *
     * @param Event $event -- The event object contains relevant data for payment notification, including Raw Payload, GET data, POST data, server variables, and a variable representing an instance with two methods.
     * @since 1.0.0
     */

    public function onPaymentNotify(Event $event)
    {
        /**
         * Extracting payment-related notification data from the event arguments.
         * If the subject is not set in the arguments, a default empty object is used.
         */
        $arguments         = $event->getArguments();
        $paymentNotifyData = $arguments['subject'] ?: new \stdClass();

        $constants = new PayKeeperConstants();
        $pk_secret = $constants->getSecret();

        // Raw payload from the webhook provider
        $rawPayload        = $paymentNotifyData->raw_payload;
        // Array of Get Parameters/Data from the webhook provider
        $getParameters     = $paymentNotifyData->get_data;
        // Array of Post Data from the webhook provider
        $postData          = $paymentNotifyData->post_data;
        // Array of Server Variables
        $serverVariables   = $paymentNotifyData->server_variables;
        // Represents an instance with two methods: `updateOrder` and `onOrderPlacementCompletion`.
        $order             = $paymentNotifyData->order;

        /**
         * Now, proceed with the necessary steps to verify the validity of the notification and ensure that it originated
         * from your payment system and retrieve `order_id`, `payment_status` and `transactionID`.
         *
         * Map the payment statuses to our system's supported statuses, which include "paid," "pending," "failed," and
         * "canceled." Additionally, retrieve the error reasons for payments marked as failed or canceled.
         */

        try
        {
            if ($serverVariables['REQUEST_METHOD'] !== 'POST') {
                echo 'Unavailable request';
                exit;
            }
            $transaction_id = $postData['id'];
            $order_id = $postData['orderid'];
            $client_id = $postData['clientid'];
            $secret = $postData['key'];
            $sum = $postData['sum'];

            $data = (object) [
                'id'                   => $order_id, // Order ID
                'payment_status'       => 'paid', // Payment Status
                'payment_error_reason' => null,  // Add reason if payment status is failed or canceled
                'transaction_id'       => $transaction_id, // Transaction ID
            ];

            // This method handles the tasks of updating the order and payment status, as well as sending emails based on the payment status.
            $order->updateOrder($data);

            // If the payment status is 'paid', utilize the `onOrderPlacementCompletion` method. This method is responsible for reducing the quantity from the inventory and removing the cart items.
            $order->onOrderPlacementCompletion();

            $sum_format = number_format($sum, 2, '.', '');
            $hash = md5($transaction_id . $sum_format . $client_id . $order_id . $pk_secret);
            if (!hash_equals($hash, $secret)) {
                echo 'Hash mismatch';
                exit;
            }

            echo 'OK ' . md5($transaction_id . $pk_secret);

        } catch (\Throwable $error) {
            Log::add($error->getMessage(), Log::ERROR, 'paykeeper.easystore');
            echo $error->getMessage();
        }
        exit;
    }
}
