<?php

use Tygh\Enum\OrderStatuses;

if (!defined('BOOTSTRAP')) { die('Access denied'); }

/**
 * Deletes all records associated with the "paykeeper.php" payment processor.
 *
 * This function removes data on payment processors, payments, and payment descriptions
 * associated with the "paykeeper.php" processor from the relevant database tables.
 *
 * @return void
 */
function fn_paykeeper_delete_payment_processors()
{
    db_query("DELETE FROM ?:payment_descriptions WHERE payment_id IN (SELECT payment_id FROM ?:payments WHERE processor_id IN (SELECT processor_id FROM ?:payment_processors WHERE processor_script = 'paykeeper.php'))");
    db_query("DELETE FROM ?:payments WHERE processor_id IN (SELECT processor_id FROM ?:payment_processors WHERE processor_script = 'paykeeper.php')");
    db_query("DELETE FROM ?:payment_processors WHERE processor_script = 'paykeeper.php'");
}

/**
 * Retrieves the tax rate for a specific item within an order based on item type.
 *
 * This function searches through order tax information to find the applicable tax rate
 * for a given item, considering its type (e.g., product, shipping, or payment).
 *
 * @param array $order_info_taxes Order's tax information.
 * @param mixed $item_id ID of the item for which tax rate is being retrieved.
 * @param string $type Type of the item (e.g., "product", "shipping", "payment").
 * @return int|null Tax rate for the specified item, or null if not applicable.
 */
function fn_paykeeper_get_taxrate($order_info_taxes, $item_id, $type)
{
    $priority = null;
    $tax_rate = null;
    $itemId = $item_id;
    foreach ($order_info_taxes as $current_tax) {
        $current_tax_items = array();

        switch ($type) {
            case "product":
                if (isset($current_tax["applies"]["items"]["P"])) {
                    $current_tax_items = $current_tax["applies"]["items"]["P"];
                }
                break;

            case "shipping":
                if (isset($current_tax["applies"]["items"]["S"][0])) {
                    $current_tax_items = $current_tax["applies"]["items"]["S"][0];
                }
                break;

            case "payment":
                $tax = db_get_row("SELECT a.priority as a_priority, b.rate_value as b_rate_value FROM ?:taxes as a LEFT JOIN ?:tax_rates as b ON b.tax_id = a.tax_id WHERE a.tax_id = ?i", (int)$current_tax);
                if (isset($tax['a_priority'], $tax['b_rate_value'])) {
                    $current_tax = [];
                    $current_tax['priority'] = $tax['a_priority'];
                    $current_tax['rate_value'] = $tax['b_rate_value'];
                    $current_tax_items[] = $tax['b_rate_value'];
                }
                break;
        }

        if (empty($current_tax_items)) {
            continue;
        }

        foreach ($current_tax_items as $key => $value) {
            if ($itemId != $key || !$value) {
                continue;
            }

            if (is_null($priority) || $priority > $current_tax['priority']) {
                $priority = $current_tax['priority'];
                $tax_rate = (int)$current_tax["rate_value"];
            }
        }
    }

    return $tax_rate;
}

/**
 * Formats a given price to two decimal places in a standard format required by PayKeeper.
 *
 * This function ensures that the price is always displayed with two decimal places,
 * such as "8.00" or "15.50", which is necessary for PayKeeper’s requirements.
 *
 * @param float $price Price value to be formatted.
 * @return string Formatted price as a string with two decimal places.
 */
function fn_paykeeper_format_price($price)
{
    // allows 8 or 8.00 formats only
    return number_format(fn_format_price($price), 2, '.', '');
}

/**
 * Checks if gift certificates are applied to an order.
 *
 * This function determines if any gift certificates have been used for a specified order.
 * If the order is a parent order (i.e., it has sub-orders), the function also checks each sub-order
 * for the use of gift certificates.
 *
 * @param array $order_info Information about the order, including any gift certificate usage.
 * @return bool True if a gift certificate is used; false otherwise.
 */
function fn_paykeeper_use_gift_certificates($order_info)
{
    $use_gift_certificates = !empty($order_info['use_gift_certificates']);

    $is_parent_order = (class_exists('\Tygh\Enum\OrderStatuses')
        ? ($order_info['status'] === OrderStatuses::PARENT)
        : ($order_info['status'] === 'P')
    );

    if (!$use_gift_certificates && $is_parent_order) {
        $order_ids = db_get_fields('SELECT order_id FROM ?:orders WHERE parent_order_id = ?i', $order_info['order_id']);

        foreach ($order_ids as $order_id) {
            $order_info = fn_get_order_info($order_id);

            if (!empty($order_info['use_gift_certificates'])) {
                $use_gift_certificates = true;
                break;
            }
        }
    }

    return $use_gift_certificates;
}

function fn_paykeeper_get_payment_form ($pk_obj, $redirect_form, $no_delay = false) {
    $to_hash = $pk_obj->getOrderTotal(true) .
        $pk_obj->getOrderParams('clientid') .
        $pk_obj->getOrderParams('orderid') .
        $pk_obj->getOrderParams('service_name') .
        $pk_obj->getOrderParams('client_email') .
        $pk_obj->getOrderParams('client_phone') .
        $pk_obj->getOrderParams('secret_key');
    $sign = hash ('sha256' , $to_hash);

    $form = "";
    $delay = ($no_delay) ? 0 : 2000;
    if ($pk_obj->getPaymentFormType() == "create") { //create form
        $form = '
                <h3>Сейчас Вы будете перенаправлены на страницу банка.</h3>
                <form name="pay_form" id="pay_form" action="'.$pk_obj->getOrderParams("form_url").'" accept-charset="utf-8" method="post">
                <input type="hidden" name="sum" value = "'.$pk_obj->getOrderTotal().'"/>
                <input type="hidden" name="orderid" value = "'.$pk_obj->getOrderParams("orderid").'"/>
                <input type="hidden" name="clientid" value = "'.$pk_obj->getOrderParams("clientid").'"/>
                <input type="hidden" name="client_email" value = "'.$pk_obj->getOrderParams("client_email").'"/>
                ';
        if (!empty($pk_obj->getOrderParams("receipt_properties"))) {
            $form .= '
            <input type="hidden" name="receipt_properties" value = \''.$pk_obj->getOrderParams("receipt_properties").'\'/>
            ';
        }
        if (!empty($pk_obj->getOrderParams("user_result_callback"))) {
            $form .= '
            <input type="hidden" name="user_result_callback" value = \''.$pk_obj->getOrderParams("user_result_callback").'\'/>
            ';
        }
        $form .= '
                <input type="hidden" name="client_phone" value = "'.$pk_obj->getOrderParams("client_phone").'"/>
                <input type="hidden" name="service_name" value = "'.$pk_obj->getOrderParams("service_name").'"/>
                <input type="hidden" name="cart" value = \''.htmlentities($pk_obj->getFiscalCartEncoded(),ENT_QUOTES).'\' />
                <input type="hidden" name="sign" value = "'.$sign.'"/>
                <input type="hidden" name="lang" value = "'.$pk_obj->getCurrentLang().'"/>
                <input type="submit" class="btn btn-default" value="Оплатить"/>
                </form>';

        if ($redirect_form) {
            $form .= '
            <script type="text/javascript">
                window.addEventListener("load", submitPayForm);
                function submitPayForm() {
                    setTimeout(function() {
                        document.forms["pay_form"].submit();
                    }, '.$delay.');
                }
                </script>
            ';
        }
    }
    else { //order form
        $payment_parameters = array(
            "clientid"=>$pk_obj->getOrderParams("clientid"),
            "orderid"=>$pk_obj->getOrderParams('orderid'),
            "sum"=>$pk_obj->getOrderTotal(),
            "client_phone"=>$pk_obj->getOrderParams("phone"),
            "phone"=>$pk_obj->getOrderParams("phone"),
            "client_email"=>$pk_obj->getOrderParams("client_email"),
            "cart"=>$pk_obj->getFiscalCartEncoded());
        $query = http_build_query($payment_parameters);
        if( function_exists( "curl_init" )) { //using curl
            $CR = curl_init();
            curl_setopt($CR, CURLOPT_URL, $pk_obj->getOrderParams("form_url"));
            curl_setopt($CR, CURLOPT_POST, 1);
            curl_setopt($CR, CURLOPT_FAILONERROR, true);
            curl_setopt($CR, CURLOPT_POSTFIELDS, $query);
            curl_setopt($CR, CURLOPT_RETURNTRANSFER, 1);
            curl_setopt($CR, CURLOPT_SSL_VERIFYPEER, 0);
            $result = curl_exec( $CR );
            $error = curl_error( $CR );
            if( !empty( $error )) {
                $form = "<br/><span class=message>"."INTERNAL ERROR:".$error."</span>";
            }
            else {
                $form = $result;
            }
            curl_close($CR);
        }
        else { //using file_get_contents
            if (!ini_get('allow_url_fopen')) {
                $form = "<br/><span class=message>"."INTERNAL ERROR: Option allow_url_fopen is not set in php.ini"."</span>";
            }
            else {
                $query_options = array("https"=>array(
                    "method"=>"POST",
                    "header"=>
                        "Content-type: application/x-www-form-urlencoded",
                    "content"=>$query
                ));
                $context = stream_context_create($query_options);
                $form = file_get_contents($pk_obj->getOrderParams("form_url"), false, $context);
            }
        }
    }
    if ($form  == "") {
        $form = '<h3>Произошла ошибка при инциализации платежа</h3>';
    }

    return $form;
}