<?php

//namespace Bitrix\Modules\Sale\Handler\Paysystem\Paykeeper;
//namespace Sale\Handlers\PaySystem\Paykeeper;

use \Bitrix\Sale\Order;
use \Bitrix\Sale\PaySystem;
use \Bitrix\Main\Web\HttpClient;
use \Bitrix\Main\Web\Json;
use \Bitrix\Main\Event;
use \Bitrix\Main\EventResult;
use \Bitrix\Main\EventManager;
use \Bitrix\Main\Loader;
use \Bitrix\Main\Context;

use \Bitrix\Main\Application;
use \Bitrix\Main\Entity;

use \Bitrix\Sale\Internals\PaymentTable;
use \Bitrix\Sale\Internals\PayableItemTable;

//use Sale\Handlers\PaySystem\Paykeeper\Lib\PayKeeperPayment;
//use Sale\Handlers\PaySystem\Paykeeper\Lib\PayKeeperApiService;
//use Sale\Handlers\PaySystem\Paykeeper\Lib\Utils;
//use Sale\Handlers\PaySystem\Paykeeper\Lib\PayKeeperCart;
//use Sale\Handlers\PaySystem\Paykeeper\Lib\Db\OrdersTable;

EventManager::getInstance()->addEventHandler(
    'sale',
    'OnSaleStatusOrderChange',
    [__NAMESPACE__ . '\PaykeeperEventHandler','OnSaleStatusOrderSaved']
);

include_once __DIR__ . '/lib/Utils.php';
include_once __DIR__ . '/lib/PayKeeperApiService.php';
require_once __DIR__ . '/lib/PayKeeperPayment.php';
require_once __DIR__ . '/lib/PayKeeperCart.php';
require_once __DIR__ . '/lib/db/OrdersTable.php';

/**
 * Class PaykeeperEventHandler
 * Handles events and actions related to Paykeeper integration.
 */
class PaykeeperEventHandler
{
    private static $logging = false;
    private static $errorData = [];
    private static $paramsLog = [];
    private static $paySystemId;
    private static $personTypeId;
    private static $paykeeperUrl;
    /**
     * @var PaykeeperPayment
     */
    private static $pkObj;

    /**
     * Event handler for the OnSaleStatusOrderSaved event.
     *
     * @param $event
     * @return EventResult
     */
    public static function OnSaleStatusOrderSaved($event)
    {
        $order = $event->getParameter("ENTITY");
//        $oldValues = $event->getParameter("VALUES");
//        if (!isset($oldValues['STATUS_ID'])) {
//            return self::getSuccessEventResult();
//        }

        self::$paySystemId = $order->getField('PAY_SYSTEM_ID');
        self::$personTypeId = $order->getField('PERSON_TYPE_ID');

        // Получаем ID платежа РК
        $paymentCollection = $order->getPaymentCollection();
        $payment = null;
        foreach ($paymentCollection as $paymentElement) {
            if ($paymentElement->getField('PAY_SYSTEM_ID') != self::$paySystemId
                || !Utils::isThisSystemID(self::$paySystemId))
            {
                continue;
            }

            $payment = $paymentElement;
            break;
        }

        if (!$payment) {
            return self::getSuccessEventResult();
        }

        $orderAmount = $order->getPrice();
        $paymentAmount = $payment->getField('PS_SUM');

        Utils::paykeeperLogger([
            'TITLE' => 'EVENT_START',
            'MESSAGE' => __METHOD__ . ':' . __LINE__,
            'PAY_SYSTEM_ID' => self::$paySystemId,
            'PERSON_TYPE_ID' => self::$personTypeId,
            'ORDER_SUM' => $orderAmount,
            'PAYMENT_SUM' => $paymentAmount,
            'ORDER_IS_PAID' => $order->isPaid() ? 'Y' : 'N',
        ]);

        if ($orderAmount == $paymentAmount && (!$order->isPaid() || $payment->getField('PS_STATUS') === 'Y')) {
            return self::getSuccessEventResult();
        }

        $paykeeper_hold_funds_enable = self::getBusVal('PAYKEEPER_HOLD_FUNDS_ENABLE') === 'Y';
        $paykeeper_print_final_receipt = self::getBusVal('PAYKEEPER_PRINT_FINAL_RECEIPT') === 'Y';
        $paykeeper_charge_part_enable = self::getBusVal('PAYKEEPER_CHARGE_PART_ENABLE') === 'Y';
        $paykeeper_login = self::getBusVal('PAYKEEPER_LK_LOGIN');
        $paykeeper_password = self::getBusVal('PAYKEEPER_LK_PASSWORD');
        self::$paykeeperUrl = Utils::clearPaykeeperUrl(self::getBusVal('PAYKEEPER_FORM_URL'));
        self::$logging = (self::getBusVal('PAYKEEPER_LOGGING') === 'Y');

        if (!self::$paykeeperUrl || !$paykeeper_login || !$paykeeper_password
            || (!$paykeeper_print_final_receipt && !$paykeeper_hold_funds_enable))
        {
            return self::getSuccessEventResult();
        }

        $paykeeper_hold_debit_after_status = self::getBusVal('PAYKEEPER_HOLD_DEBIT_AFTER_STATUS');
        $paykeeper_hold_cancel_after_status = self::getBusVal('PAYKEEPER_HOLD_CANCEL_AFTER_STATUS');
        $paykeeper_print_after_status = self::getBusVal('PAYKEEPER_PRINT_AFTER_STATUS');

        $chargeSuccessAfterStatus = self::getBusVal('PAYKEEPER_CHARGE_SUCCESS_AFTER_STATUS');
        $chargeFailAfterStatus = self::getBusVal('PAYKEEPER_CHARGE_FAIL_AFTER_STATUS');

        $order_status_id = $order->getField('STATUS_ID');

        $paykeeper_is_receipt = $paykeeper_print_final_receipt
            && $paykeeper_print_after_status == $order_status_id;
        $paykeeper_is_hold_debit = $paykeeper_hold_funds_enable && !$paykeeper_charge_part_enable
            && $paykeeper_hold_debit_after_status == $order_status_id;
        $paykeeper_is_hold_cancel = $paykeeper_hold_funds_enable
            && $paykeeper_hold_cancel_after_status == $order_status_id;
        $paykeeper_is_charge_part = $paykeeper_hold_funds_enable && $paykeeper_charge_part_enable
            && $paykeeper_hold_debit_after_status === $order_status_id;

        if (!$paykeeper_is_receipt
            && !$paykeeper_is_hold_debit
            && !$paykeeper_is_hold_cancel
            && !$paykeeper_is_charge_part)
        {
            return self::getSuccessEventResult();
        }

        $orderId = $order->getId();
        $invoiceId = $payment->getField('PS_INVOICE_ID');

        if ($paykeeper_is_charge_part) {
            OrdersTable::createTable();
        }

        $PayKeeperApiService = new PayKeeperApiService(self::$paykeeperUrl, $paykeeper_login, $paykeeper_password);

        /** @var \Sale\Handlers\PaySystem\PayKeeperHandler $service */
        $service = PaySystem\Manager::getObjectById(self::$paySystemId);

        // Hold charge (classic)
        if ($paykeeper_is_hold_debit) {
            $resultConfirm = $service->confirm($payment);
            self::$paramsLog[] = [
                'SEVERITY' => 'INFO',
                'AUDIT_TYPE_ID' => 'Списание средств по ранее проведённой авторизации PayKeeper',
                'MODULE_ID' => 'sale',
                'ITEM_ID' => "API запрос /change/payment/capture/",
                'DESCRIPTION' => "Заказ $orderId. Списание средств!"
            ];
            if (!$resultConfirm->isSuccess()) {
                self::$errorData[] = [
                    'event_code' => 'CAPTURE',
                    'event_name' => 'onHoldDebitPaykeeper',
                    'log_name' => 'СПИСАНИЕ СРЕДСТВ',
                    'log_info' => "Заказ $orderId. Ошибка: " . implode(', ', $resultConfirm->getErrorMessages())
                ];
            }
        }

        // Payment hold operations (partial refund, full capture, or cancellation with recurring charge)
        if ($paykeeper_is_charge_part) {
            // Определяем сценарий списания
            $actionPayment = self::determineActionPayment($orderAmount, $paymentAmount);

            Utils::paykeeperLogger([
                'TITLE' => 'INIT_ACTION_PAYMENT',
                'ACTION_PAYMENT' => $actionPayment
            ], self::$logging);

            switch ($actionPayment) {
                case 'HOLD_PARTIAL_REFUND': // Сумма заказа < суммы холда: частичный возврат со списанием
                    $refundableSum = $paymentAmount - $orderAmount;
                    $resultRefund = $service->refund($payment, $refundableSum);
                    self::$paramsLog[] = [
                        'SEVERITY' => 'INFO',
                        'AUDIT_TYPE_ID' => 'Частичный возврат со списанием средств по ранее проведённой авторизации PayKeeper',
                        'MODULE_ID' => 'sale',
                        'ITEM_ID' => 'API запрос /change/payment/reverse/',
                        'DESCRIPTION' => "Заказ $orderId. Частичный возврат со списанием средств!",
                    ];

                    if (!$resultRefund->isSuccess()) {
                        self::$errorData[] = [
                            'event_code' => 'PARTCAPTURE',
                            'event_name' => 'onHoldPartDebitPaykeeper',
                            'log_name' => 'ЧАСТИЧНОЕ СПИСАНИЕ СРЕДСТВ',
                            'log_info' => "Заказ $orderId. Списание средств!"
                        ];
                        if ($chargeFailAfterStatus) {
                            $order = Order::load($payment->getOrderId());
                            $order->setFieldNoDemand('STATUS_ID', $chargeFailAfterStatus);
                            $order->save();
                        }
                    } else {
                        if ($chargeSuccessAfterStatus) {
                            $order = Order::load($payment->getOrderId());
                            $order->setFieldNoDemand('STATUS_ID', $chargeSuccessAfterStatus);
                            $order->save();
                        }
                    }
                    break;

                case 'HOLD_FULL_CAPTURE': // Сумма заказа = сумме холда: полное списание по холду
                    $resultConfirm = $service->confirm($payment);
                    self::$paramsLog[] = [
                        'SEVERITY' => 'INFO',
                        'AUDIT_TYPE_ID' => 'Списание средств по ранее проведённой авторизации PayKeeper',
                        'MODULE_ID' => 'sale',
                        'ITEM_ID' => 'API запрос /change/payment/capture/',
                        'DESCRIPTION' => "Заказ $orderId. Списание средств!"
                    ];
                    if (!$resultConfirm->isSuccess()) {
                        self::$errorData[] = [
                            'event_code' => 'CAPTURE',
                            'event_name' => 'onHoldDebitPaykeeper',
                            'log_name' => 'СПИСАНИЕ СРЕДСТВ',
                            'log_info' => "Заказ $orderId. Ошибка: " . implode(', ', $resultConfirm->getErrorMessages())
                        ];
                    }
                    break;

                case 'HOLD_CANCEL_WITH_RECURRING': // Сумма заказа > суммы холда
                    $resultCancel = $service->cancel($payment);
                    $errorFlag = false;
                    self::$paramsLog[] = [
                        'SEVERITY' => 'INFO',
                        'AUDIT_TYPE_ID' => 'Отмена авторизации средств PayKeeper',
                        'MODULE_ID' => 'sale',
                        'ITEM_ID' => 'API запрос /change/payment/reverse/',
                        'DESCRIPTION' => "Заказ $orderId. Отмена авторизации средств PayKeeper"
                    ];
                    if (!$resultCancel->isSuccess()) {
                        self::$errorData[] = [
                            'event_code' => 'REVERSE',
                            'event_name' => 'onHoldCancelPaykeeper',
                            'log_name' => 'ОТМЕНА АВТОРИЗАЦИИ',
                            'log_info' => "Заказ $orderId. Ошибка: " . implode(', ', $resultCancel->getErrorMessages())
                        ];
                        $errorFlag = true;
                    } else {
                        $resultRecurrent = $service->repeatRecurrent($payment);
                        self::$paramsLog[] = [
                            'SEVERITY' => 'INFO',
                            'AUDIT_TYPE_ID' => 'Безакцептное списание средств PayKeeper',
                            'MODULE_ID' => 'sale',
                            'ITEM_ID' => 'API запрос /change/binding/execute/',
                            'DESCRIPTION' => "Заказ $orderId. Безакцептное списание средств PayKeeper"
                        ];
                        if (!$resultRecurrent->isSuccess()) {
                            self::$errorData[] = [
                                'event_code' => 'BINDING',
                                'event_name' => 'onHoldBindingPaykeeper',
                                'log_name' => 'БЕЗАКЦЕПТНОЕ СПИСАНИЕ',
                                'log_info' => "Заказ $orderId. Ошибка: " . implode(', ', $resultRecurrent->getErrorMessages())
                            ];
                            $errorFlag = true;
                        }
                    }
                    if ($errorFlag) {
                        if ($chargeFailAfterStatus) {
                            $order = Order::load($payment->getOrderId());
                            $order->setFieldNoDemand('STATUS_ID', $chargeFailAfterStatus);
                            $order->save();
                        }
                    } else {
                        if ($chargeSuccessAfterStatus) {
                            $order = Order::load($payment->getOrderId());
                            $order->setFieldNoDemand('STATUS_ID', $chargeSuccessAfterStatus);
                            $order->save();
                        }
                    }
                    break;
            }
        }

        // Event Print Receipt
        if ($paykeeper_is_receipt) {
            self::generateFinalReceipt($invoiceId, $order, $PayKeeperApiService);
        }

        // Event Hold Cancel
        if ($paykeeper_is_hold_cancel) {
            $resultCancel = $service->cancel($payment);
            if (!$resultCancel->isSuccess()) {
                self::$errorData[] = [
                    'event_code' => 'REVERSE',
                    'event_name' => 'onHoldCancelPaykeeper',
                    'log_name' => 'ОТМЕНА АВТОРИЗАЦИИ',
                    'log_info' => "Заказ $orderId. Ошибка: " . implode(', ', $resultCancel->getErrorMessages())
                ];
            }
            self::$paramsLog[] = [
                'SEVERITY' => 'INFO',
                'AUDIT_TYPE_ID' => 'Отмена авторизации средств PayKeeper',
                'MODULE_ID' => 'sale',
                'ITEM_ID' => 'API запрос /change/payment/reverse/',
                'DESCRIPTION' => "Заказ $orderId. Отмена авторизации средств PayKeeper"
            ];
        }

        // Добавляем запись в журнал событий об API-запросе
        if (!empty(self::$paramsLog)) {
            foreach (self::$paramsLog as $log) {
                \CEventLog::Add($log);
            }
        }

        // Добавляем всплывающее уведомление в административной панели
        if (!empty(self::$errorData)) {
            foreach (self::$errorData as $error) {
                self::triggerPaykeeperEventError($error['log_info'], $orderId, $error['event_name']);

                \CAdminNotify::Add([
                    'TAG' => "PAYKEEPER_{$orderId}_{$error['event_code']}",
                    'MESSAGE' => "{$error['log_name']} PAYKEEPER. {$error['log_info']}",
                    'NOTIFY_TYPE' => 'E'
                ]);
            }
        }

        return self::getSuccessEventResult();
    }

    /**
     * Инициирование генерации чека окончательного расчёта
     *
     * @param $invoiceId
     * @param $order
     * @param PayKeeperApiService $PayKeeperApiService
     * @return void
     */
    private static function generateFinalReceipt($invoiceId, $order, $PayKeeperApiService)
    {
        $orderId = $order->getId();
        // Создаем новую группу свойств и свойство для чека окончательного расчета, если их нет
        $propertyCode = 'PAYKEEPER_RECEIPT_ID';
        $propertyName = 'Номер запроса на генерацию чека в PK';
        self::createOrderPropertyForAllPersonTypes($propertyName, $propertyCode);
        $requestData = [ 'id' => $invoiceId ];
        $paykeeperUri = '/change/payment/post-sale-receipt/';
        $sendResult = $PayKeeperApiService::receiptPayment($invoiceId);

        Utils::paykeeperLogger([
            'TITLE' => 'PAYMENT_RECEIPT',
            'URL' => self::$paykeeperUrl . $paykeeperUri,
            'METHOD' => 'POST',
            'DATA' => $requestData,
            'RESPONSE' => $sendResult
        ], self::$logging);

        $sendResult = isset($sendResult[0]) ? $sendResult[0] : $sendResult;
        $errorFlag = false;
        if (isset($sendResult['receipt_id'])) {
            $receipt_id = $sendResult['receipt_id'];
            $infoLog = "Заказ $orderId. ID чека $receipt_id";

            // Записываем значение номера запроса печати чека
            /** @var \Bitrix\Sale\PropertyValueCollection $propertyCollection */
            $propertyCollection = $order->getPropertyCollection();
            $property = $propertyCollection->getItemByOrderPropertyCode($propertyCode);

            if($property && $property->getValue() != $receipt_id) {
                $property->setField('VALUE', $receipt_id);
                $order->save();
            }
        } else if (isset($sendResult['msg'])) {
            $errorFlag = true;
            $infoLog = "Заказ $orderId. Ошибка: {$sendResult['msg']}";
        } else {
            $errorFlag = true;
            $infoLog = "Заказ $orderId. Неизвестная ошибка!";
        }

        if ($errorFlag) {
            self::$errorData[] = [
                'event_code' => 'RECEIPT',
                'event_name' => 'onPrintCheckPaykeeper',
                'log_name' => 'ГЕНЕРАЦИЯ ЧЕКА',
                'log_info' => $infoLog
            ];
        }

        self::$paramsLog[] = [
            'SEVERITY' => 'INFO',
            'AUDIT_TYPE_ID' => 'Печать чека окончательного расчёта PayKeeper',
            'MODULE_ID' => 'sale',
            'ITEM_ID' => "API запрос $paykeeperUri",
            'DESCRIPTION' => $infoLog,
        ];
    }

    /**
     * Определение сценария списания
     *
     * @param $orderAmount
     * @param $holdAmount
     * @return string
     */
    private static function determineActionPayment($orderAmount, $holdAmount)
    {
        if ($orderAmount < $holdAmount) {
            return 'HOLD_PARTIAL_REFUND';
        } elseif ($orderAmount == $holdAmount) {
            return 'HOLD_FULL_CAPTURE';
        } else {
            return 'HOLD_CANCEL_WITH_RECURRING';
        }
    }

    /**
     * Creates an order property for all person types.
     *
     * This method checks if a property with a specific code exists for each person type in the system.
     * If not, it creates the property and assigns it to a property group.
     *
     * @param string $propertyName The name of the property to be created.
     * @param string $propertyCode The unique code of the property.
     * @return void
     */
    private static function createOrderPropertyForAllPersonTypes($propertyName, $propertyCode)
    {
        // Получаем все типы плательщиков
        $personTypes = [];
        $dbPersonTypes = \CSalePersonType::GetList( ['SORT' => 'ASC'], [] );
        while ($personType = $dbPersonTypes->Fetch()) {
            $personTypes[] = $personType['ID'];
        }

        foreach ($personTypes as $personTypeId) {
            // Проверяем, существует ли уже свойство с таким кодом для данного типа плательщика
            $dbProps = \CSaleOrderProps::GetList(
                [],
                [ 'CODE' => $propertyCode, 'PERSON_TYPE_ID' => $personTypeId ]
            );

            if ($dbProps->Fetch())
                continue;

            // Создаем новую группу свойства
            $groupExists = false;
            $propsGroupId = null;
            $nameGroup = 'Генерация чека окончательного расчета';
            $groupCheck = \CSaleOrderPropsGroup::GetList( [],
                [ 'NAME' => $nameGroup, 'PERSON_TYPE_ID' => $personTypeId ]
            );

            if ($group = $groupCheck->Fetch()) {
                $groupExists = true;
                $propsGroupId = $group['ID'];
            }

            if (!$groupExists) {
                $groupFields = [
                    'NAME' => $nameGroup,
                    'PERSON_TYPE_ID' => $personTypeId,
                    'SORT' => 500
                ];
                $propsGroupId = \CSaleOrderPropsGroup::Add($groupFields);
            }

            // Создаем новое свойство
            $propertyFields = [
                'NAME' => $propertyName,
                'TYPE' => 'STRING', // Тип свойства: STRING, NUMBER, FILE, etc.
                'REQUIRED' => 'N', // Обязательное: Y или N
                'DEFAULT_VALUE' => '',
                'SORT' => 500,
                'USER_PROPS' => 'Y', // Включить в свойства пользователя: Y или N
                'IS_LOCATION' => 'N',
                'IS_LOCATION4TAX' => 'N',
                'IS_EMAIL' => 'N',
                'IS_PROFILE_NAME' => 'N',
                'IS_PAYER' => 'N',
                'IS_FILTERED' => 'N',
                'CODE' => $propertyCode,
                'ACTIVE' => 'Y',
                'UTIL' => 'Y',
                'INPUT_FIELD_LOCATION' => '',
                'PROPS_GROUP_ID' => $propsGroupId, // ID группы свойств
                'SIZE1' => 0,
                'SIZE2' => 0,
                'DESCRIPTION' => '',
                'PERSON_TYPE_ID' => $personTypeId // Тип плательщика
            ];
            \CSaleOrderProps::Add($propertyFields);
        }
    }

    /**
     * Triggers an event in case of an error in the Paykeeper API request.
     *
     * This method logs the error message and triggers the custom event 'onPrintCheckPaykeeper'.
     * Additional error handling can be performed in the event handler.
     *
     * @param string $errMessage The error message received from the Paykeeper API.
     * @param string $orderId The ID of the order that encountered the error.
     * @param string $nameEvent
     * @return void
     */
    private static function triggerPaykeeperEventError($errMessage, $orderId, $nameEvent) {
        $event = new Event('paykeeper', $nameEvent, [ $errMessage, $orderId ]);
        $event->send();

        foreach ($event->getResults() as $eventResult) {
            if ($eventResult->getType() == EventResult::ERROR && self::$logging) {
                $errorParams = $eventResult->getParameters();
                $log_params = [
                    'TITLE' => 'EVENT_ERROR',
                    'EVENT NAME' => $nameEvent,
                    'METHOD' => 'EVENT',
                    'DATA' => "ORDER ID $orderId MESSAGE $errMessage"
                ];
                if (!empty($errorParams['ERROR_MESSAGE'])) {
                    $log_params['TEXT ERROR'] = $errorParams['ERROR_MESSAGE'];
                }
                Utils::paykeeperLogger($log_params, self::$logging);
            }
        }
    }

    /**
     * Returns a successful event result.
     *
     * This method creates and returns an instance of `EventResult` with a success status.
     *
     * @return \Bitrix\Main\EventResult An event result object with a status of success.
     */
    private static function getSuccessEventResult() {
        return new EventResult(EventResult::SUCCESS);
    }

    /**
     * Retrieves the business value associated with a specified parameter for a given order.
     *
     * @param string $param The name of the business parameter to retrieve.
     * @param \Bitrix\Sale\Order $order The order object containing relevant payment system and person type IDs.
     *
     * @return mixed Returns the business value associated with the specified parameter for the payment system
     *               and person type in the order.
     */
    private static function getBusVal($param)
    {
        return \Bitrix\Sale\BusinessValue::get(
            $param,
            PaySystem\Service::PAY_SYSTEM_PREFIX . self::$paySystemId,
            self::$personTypeId
        );
    }

    /**
     * Sends a request to the specified URL using cURL or HttpClient depending on availability.
     *
     * @param string $url The URL to send the request to.
     * @param array $data The data to send with the request.
     * @param array $headers Optional. Additional headers to include in the request. Default is an empty array.
     * @param string $datatype Optional. The format of the data, such as 'json'. Default is 'json'.
     * @return array|mixed
     */
    private static function Send($url, $data, $headers = [], $datatype = 'json')
    {

        global $APPLICATION;

        if (mb_strtoupper(SITE_CHARSET) != 'UTF-8') {
            $data = $APPLICATION->ConvertCharsetArray($data, 'windows-1251', 'UTF-8');
        }

        if(function_exists('curl_init')) {
            $response = self::SendCurl($url, $data, $headers, $datatype);
        } else {
            $response = self::SendHttpClient($url, $data, $headers, $datatype);
        }

        return $response;
    }

    /**
     * Sends an HTTP request using the HttpClient class.
     *
     * @param string $url The URL to send the request to.
     * @param array $data The data to send with the request. Can be null for GET requests.
     * @param array $headers Additional headers to include in the request.
     * @param string $datatype The format of the response data, e.g., 'json'. Default is 'json'.
     * @return array
     */
    private static function SendHttpClient($url, $data, $headers, $datatype)
    {

        global $APPLICATION;

        $httpClient = new HttpClient();
        $httpClient->setCharset('utf-8');
        $httpClient->disableSslVerification();
        foreach ($headers as $param => $value) {
            $httpClient->setHeader($param, $value);
        }
        if ($data) {
            $httpClient->post($url, $data);
        } else {
            $httpClient->get($url);
        }

        $response =  $httpClient->getResult();
        if ($datatype == 'json' && self::isJson($response)) {
            $response =  Json::decode($response);
        } else if (empty($httpClient->getError())) {
            $response = [
                'status' => 'success',
                'msg' => $response
            ];
        } else {
            $response = [
                'errorCode' => 999,
                'errorMessage' => 'Server not available',
            ];
        }

        if (mb_strtoupper(SITE_CHARSET) != 'UTF-8') {
            $APPLICATION->ConvertCharsetArray($response, 'UTF-8', 'windows-1251');
        }

        return $response;
    }

    /**
     * Sends an HTTP request using cURL.
     *
     * @param string $url The URL to send the request to.
     * @param array $data The data to send with the request. Can be null for GET requests.
     * @param array $headers Additional headers to include in the request.
     * @param string $datatype The format of the response data, such as 'json'. Default is 'json'.
     * @return array|mixed
     */
    private static function SendCurl($url, $data, $headers, $datatype)
    {
        $curl_opt = [
            CURLOPT_VERBOSE => true,
            CURLOPT_SSL_VERIFYHOST => false,
            CURLOPT_SSL_VERIFYPEER =>false,
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_HEADER => false
        ];
        $set_headers = [];
        foreach ($headers as $param => $value) {
            $set_headers[] = "$param: $value";
        }
        if ($set_headers) {
            $curl_opt[CURLOPT_HTTPHEADER] = $set_headers;
        }
        if ($data) {
            $curl_opt[CURLOPT_POST] = true;
            $curl_opt[CURLOPT_POSTFIELDS] = http_build_query($data);
        }
        $ch = curl_init();
        curl_setopt_array($ch, $curl_opt);
        $response = curl_exec($ch);
        $error = curl_error($ch);

        if ($datatype == 'json' && self::isJson($response)) {
            $response = json_decode($response, true);
        } else if (empty($error)) {
            $response = [
                'result' => 'success',
                'msg' => $response
            ];
        } else {
            $response = [
                'errorCode' => 999,
                'errorMessage' => curl_error($ch),
            ];
        }
        curl_close($ch);

        return $response;
    }

    /**
     * Checks if the given data is valid JSON.
     *
     * @param string $json_data The JSON data to validate.
     * @param bool $return_data Optional. If true, returns the decoded JSON data. Default is false.
     * @return bool|mixed
     */
    private static function isJson($json_data, $return_data = false)
    {
        $data = json_decode($json_data);
        return (json_last_error() == JSON_ERROR_NONE) ? ($return_data ? $data : true) : false;
    }

    private static function getCardToken($orderId)
    {
        $rncbOrderTable = OrdersTable::getRow([
            'select' => [ 'ID', 'PK_BANK_ID' ],
            'filter' => [ '=BX_ORDER_ID' => $orderId ],
            'order' => [ 'ID' => 'DESC' ],
        ]);
        Utils::paykeeperLogger([
            'TITLE' => 'GET_BANK_ID',
            'MESSAGE' => "Получение bank_id",
            'BANK_ID' => substr($rncbOrderTable['PK_BANK_ID'] ?: 0, 0, 3) . Utils::$textHiddenForLog
        ], self::$logging);
        return $rncbOrderTable['PK_BANK_ID'] ?: 0;
    }
}
