<?php

require_once('paykeeper_common_class/paykeeper.class.php');

/**
 * Controller class for Paykeeper payment extension.
 */
class ControllerExtensionPaymentPaykeeper extends Controller
{

    /**
     * Display the Paykeeper payment form on the checkout page.
     * @return string Rendered view of the Paykeeper payment form.
     */
	public function index()
    {
		$data['button_confirm'] = $this->language->get('button_confirm');

        if (isset($_SERVER['HTTPS']) && ($_SERVER['HTTPS'] == 'on' || $_SERVER['HTTPS'] == 1)
            || isset($_SERVER['HTTP_X_FORWARDED_PROTO']) && $_SERVER['HTTP_X_FORWARDED_PROTO'] == 'https'
            || $this->config->get('config_ssl'))
        {
            $ssl_use = true;
        } else {
            $ssl_use = false;
        }

        $data['url_confirm'] = $this->url->link('extension/payment/paykeeper/gopay', '', $ssl_use);

		$this->load->model('checkout/order');

		$order_info = $this->model_checkout_order->getOrder($this->session->data['order_id']);

		$payment_parameters = http_build_query(array(
			"orderid"=>$this->session->data['order_id'],
			"clientid"=>$order_info['email'],
			"sum"=>$this->currency->format($order_info['total'], $order_info['currency_code'], $order_info['currency_value'], false),
			"phone"=>$order_info['telephone']
		));

		$data['server'] = $this->config->get('paykeeperserver');
		$data['payment_parameters'] = $payment_parameters;
		
        return $this->load->view('/extension/payment/paykeeper', $data);
	}

    /**
     * Handle the callback notification from Paykeeper for a successful payment.
     * @return void
     */
    public function callback()
    {
        if ($this->request->server['REQUEST_METHOD'] == 'POST') {
            if (!isset($this->request->post['id']) || !isset($this->request->post['orderid'])) {
                echo "Error! Invalid data";
                exit;
            }

            $id = $this->request->post['id'];
            $sum = $this->request->post['sum'];
            $clientid = $this->request->post['clientid'];
            $orderid = $this->request->post['orderid'];
            $key = $this->request->post['key'];
            $secret_seed = $this->config->get('paykeepersecret');

            if (!hash_equals(md5($id . sprintf("%.2lf", $sum) . $clientid . $orderid . $secret_seed), $key)) {
                echo "Error! Hash mismatch";
                exit;
            }

            $this->load->model('checkout/order');
            $this->model_checkout_order->addOrderHistory($orderid, $this->config->get('paykeeper_order_status_id'));

            echo 'OK ' . md5($id . $secret_seed);
            exit;
        }
    }

    /**
     * Redirect to Paykeeper payment gateway.
     * @return void
     */
	public function gopay()
    {
		$this->load->language('checkout/checkout');
		$this->load->language('extension/payment/paykeeper');
		
		$data['breadcrumbs'] = array();

		$data['breadcrumbs'][] = array(
			'text' => $this->language->get('text_home'),
			'href' => $this->url->link('common/home')
		);

		$data['breadcrumbs'][] = array(
			'text' => $this->language->get('text_cart'),
			'href' => $this->url->link('checkout/cart')
		);

		$data['breadcrumbs'][] = array(
			'text' => $this->language->get('heading_title'),
			'href' => $this->url->link('checkout/checkout', '', 'SSL')
		);
		
		$data['breadcrumbs'][] = array(
			'text' => $this->language->get('paykeeper_title'),
			'href' => $this->url->link('extension/payment/paykeeper/gopay', '', 'SSL')
		);

        $data['heading_title'] = $this->language->get('paykeeper_title');
		$this->document->setTitle($data['heading_title']);
		
		$this->load->model('checkout/order');

		$data['column_left'] = $this->load->controller('common/column_left');
		$data['column_right'] = $this->load->controller('common/column_right');
		$data['content_top'] = $this->load->controller('common/content_top');
		$data['content_bottom'] = $this->load->controller('common/content_bottom');
		$data['footer'] = $this->load->controller('common/footer');
		$data['header'] = $this->load->controller('common/header');


        //generate payment form
        $data["form"] = $this->generatePaymentForm($this->session->data['order_id']);

        $this->response->setOutput($this->load->view('extension/payment/paykeeper_iframe', $data));
	}

    /**
     * Handle successful payment and clear the cart.
     * @return void
     */
	public function success()
    {
		$this->load->language('checkout/checkout');
		$this->load->language('extension/payment/paykeeper');

        //clear cart
        $this->cart->clear();
		
		$data['heading_title'] = $this->language->get('paykeeper_title');
		
		$data['breadcrumbs'] = array();

		$data['breadcrumbs'][] = array(
			'text' => $this->language->get('text_home'),
			'href' => $this->url->link('common/home')
		);

		$data['breadcrumbs'][] = array(
			'text' => $this->language->get('text_cart'),
			'href' => $this->url->link('checkout/cart')
		);

		$data['breadcrumbs'][] = array(
			'text' => $this->language->get('heading_title'),
			'href' => $this->url->link('checkout/checkout', '', 'SSL')
		);
		
		$data['message'] = $this->language->get('paykeeper_success');

		$this->document->setTitle($data['message']);
		
		$data['column_left'] = $this->load->controller('common/column_left');
		$data['column_right'] = $this->load->controller('common/column_right');
		$data['content_top'] = $this->load->controller('common/content_top');
		$data['content_bottom'] = $this->load->controller('common/content_bottom');
		$data['footer'] = $this->load->controller('common/footer');
		$data['header'] = $this->load->controller('common/header');
		
        $this->response->setOutput($this->load->view('extension/payment/paykeeper_feedback', $data));
	}

    /**
     * Handle failed payment.
     * @return void
     */
	public function failed()
    {
		$this->load->language('checkout/checkout');
		$this->load->language('extension/payment/paykeeper');
		
		$data['heading_title'] = $this->language->get('paykeeper_title');
		
		$data['breadcrumbs'] = array();

		$data['breadcrumbs'][] = array(
			'text' => $this->language->get('text_home'),
			'href' => $this->url->link('common/home')
		);

		$data['breadcrumbs'][] = array(
			'text' => $this->language->get('text_cart'),
			'href' => $this->url->link('checkout/cart')
		);

		$data['breadcrumbs'][] = array(
			'text' => $this->language->get('heading_title'),
			'href' => $this->url->link('checkout/checkout', '', 'SSL')
		);
		
		$data['message'] = $this->language->get('paykeeper_failed');

		$this->document->setTitle($data['message']);
		
		$data['column_left'] = $this->load->controller('common/column_left');
		$data['column_right'] = $this->load->controller('common/column_right');
		$data['content_top'] = $this->load->controller('common/content_top');
		$data['content_bottom'] = $this->load->controller('common/content_bottom');
		$data['footer'] = $this->load->controller('common/footer');
		$data['header'] = $this->load->controller('common/header');
		
        $this->response->setOutput($this->load->view('extension/payment/paykeeper_feedback', $data));
	}

    /**
     * Confirm the order by updating the order history.
     * @return void
     */
    public function confirm()
    {
        $this->load->model('checkout/order');
        $this->model_checkout_order->addOrderHistory($this->session->data['order_id'], 1);
    }

    /**
     * Get the the total amount of VAT rate for a given sum and tax class.
     * @param float $sum
     * @param int $tax_class_id
     * @return int
     */
    protected function getRate($sum, $tax_class_id)
    {
        $max_tax_rate = 0;
        foreach($this->tax->getRates($sum, $tax_class_id) as $tax_data) {
            if (isset($tax_data['rate']) && $tax_data['rate'] > $max_tax_rate) {
                $max_tax_rate = (int)$tax_data['rate'];
            }
        }
        return $max_tax_rate;
    }

    /**
     * For debugging
     * @param $obj_to_debug
     * @return void
     */
    protected function showDebugInfo($obj_to_debug)
    {
        echo "<pre>";
        var_dump($obj_to_debug);
        echo "</pre>";
    }

    /**
     * Generate a payment form with the order details.
     * @param int $order_id
     * @return string
     */
    public function generatePaymentForm($order_id)
    {
        $this->load->model('checkout/order');

        $deffered = false;
        if (isset($_GET['order_id'])) { // признак отложенного платежа
            $deffered = true;
            $order_id = $_GET['order_id'];

            $this->load->model('account/order');
            $this->load->model('catalog/product');

            if (!$this->customer->isLogged()) {
                $this->session->data['redirect'] = $this->url->link('account/order/info', 'order_id=' . $order_id, true);
                $this->response->redirect($this->url->link('account/login', '', true));
            }

            $order_info = $this->model_account_order->getOrder($order_id);
        } else { // признак обычного платёжа
            $this->load->model('checkout/order');
            $order_info = $this->model_checkout_order->getOrder($order_id);
        }

        //GENERATING PAYKEEPER PAYMENT FORM
        $pk_obj = new PaykeeperPayment();

        //set order params
        $pk_obj->setOrderParams(
            //sum
            $this->currency->format($order_info['total'], $order_info['currency_code'], $order_info['currency_value'], false),
            //clientid
            $order_info['firstname'] . ' ' . $order_info['lastname'],
            //orderid
            $order_info['order_id'],
            //client_email
            $order_info['email'],
            //client_phone
            $order_info['telephone'],
            //service_name
            '',
            //payment form url
            $this->config->get('paykeeperserver'),
            //secret key
            $this->config->get('paykeepersecret')
        );

        //GENERATE FZ54 CART
        $set_discount = false;
        if ($deffered) {
            $order_products = $this->model_account_order->getOrderProducts($order_id);
            $item_index = 0;
            $total_products_tax_amount = 0;
            foreach ($order_products as $product) {
                $product_tax_rate = 0;
                $product_tax_amount = 0;
                $name = $product['name'];
                $product_info = $this->model_catalog_product->getProduct($product['product_id']);
                $quantity = floatval($product['quantity']);

                if ((int)$product_info['tax_class_id'] != 0) {
                    $product_tax_rate = $this->getRate($product['price'], $product_info['tax_class_id']);
                    $product_tax_amount = $product['price'] * $product_tax_rate / 100;
                    $total_products_tax_amount += $product_tax_amount * $quantity;
                }

                $price = floatval(($product['price'] + $product_tax_amount) * $order_info['currency_value']);

                if ($quantity == 1 && $pk_obj->single_item_index < 0) {
                    $pk_obj->single_item_index = $item_index;
                }

                if ($quantity > 1 && $pk_obj->more_then_one_item_index < 0) {
                    $pk_obj->more_then_one_item_index = $item_index;
                }

                $taxes = $pk_obj->setTaxes($product_tax_rate);
                $pk_obj->updateFiscalCart(
                    $pk_obj->getPaymentFormType(),
                    $name,
                    $price,
                    $quantity,
                    0,
                    $taxes['tax']
                );

                $item_index++;
            }

            $order_totals = $this->model_account_order->getOrderTotals($order_id);
            $total_tax_amount = 0;
            $shipping_price = 0;
            $shipping_name = '';
            //add shipping parameters to cart
            foreach($order_totals as $total) {
                if ($total['code'] == 'tax') {
                    $total_tax_amount += $total['value'];
                } elseif ($total['code'] == 'shipping') {
                    $shipping_price = $total['value'];
                    $shipping_name = $total['title'];
                } elseif ($total['code'] == 'coupon' || $total['code'] == 'voucher') {
                    $set_discount = true;
                }
            }

            $shipping_tax_amount = round($total_tax_amount - $total_products_tax_amount);
            if ($shipping_price > 0) {
                $shipping_tax_rate = 0;
                if ($shipping_tax_amount > 0) {
                    $shipping_tax_rate = $shipping_tax_amount / $shipping_price * 100;
                }

                $pk_obj->setShippingPrice(floatval(($shipping_price + $shipping_tax_amount) * $order_info['currency_value']));
                $shipping_taxes = $pk_obj->setTaxes($shipping_tax_rate);

                if (!$pk_obj->checkDeliveryIncluded($pk_obj->getShippingPrice(), $shipping_name)
                    && $pk_obj->getShippingPrice() > 0)
                {
                    $shipping_name = trim(strip_tags($shipping_name));
                    $pk_obj->setUseDelivery(); //for precision correct check
                    $pk_obj->updateFiscalCart(
                        $pk_obj->getPaymentFormType(),
                        $shipping_name,
                        $pk_obj->getShippingPrice(),
                        1,
                        0,
                        $shipping_taxes['tax']
                    );
                    $pk_obj->delivery_index = count($pk_obj->getFiscalCart()) - 1;
                    $pk_obj->fiscal_cart[$pk_obj->delivery_index]['item_type'] = 'service';
                }
            }
        } else {
            $cart_data = $this->cart->getProducts();
            $item_index = 0;
            foreach ($cart_data as $item) {
                $tax_rate = 0;
                $tax_amount = 0;
                $name = $item["name"];

                if ((int)$item["tax_class_id"] != 0) {
                    $tax_rate = $this->getRate($item['price'], $item["tax_class_id"]);
                    $tax_amount = $item['price'] * $tax_rate / 100;
                }

                $price = floatval(($item['price'] + $tax_amount) * $order_info['currency_value']);
                $quantity = floatval($item['quantity']);

                if ($quantity == 1 && $pk_obj->single_item_index < 0) {
                    $pk_obj->single_item_index = $item_index;
                }

                if ($quantity > 1 && $pk_obj->more_then_one_item_index < 0) {
                    $pk_obj->more_then_one_item_index = $item_index;
                }

                $taxes = $pk_obj->setTaxes($tax_rate);
                $pk_obj->updateFiscalCart(
                    $pk_obj->getPaymentFormType(),
                    $name,
                    $price,
                    $quantity,
                    0,
                    $taxes["tax"]
                );
                $item_index++;
            }

            //add shipping parameters to cart
            if (array_key_exists('shipping_method', $this->session->data)) {
                $shipping_method = $this->session->data["shipping_method"];
                if ($shipping_method['cost'] > 0) {
                    $shipping_tax_rate = 0;
                    $shipping_tax_amount = 0;
                    $shipping_name = $shipping_method['title'];

                    if ((int)$shipping_method["tax_class_id"] != 0) {
                        $shipping_tax_rate = $this->getRate($shipping_method['cost'], $shipping_method["tax_class_id"]);
                        $shipping_tax_amount = $shipping_tax_rate * $shipping_method['cost'] / 100;
                    }

                    $pk_obj->setShippingPrice(floatval(($shipping_method['cost'] + $shipping_tax_amount) * $order_info['currency_value']));
                    $shipping_taxes = $pk_obj->setTaxes($shipping_tax_rate);

                    if (!$pk_obj->checkDeliveryIncluded($pk_obj->getShippingPrice(), $shipping_name)
                        && $pk_obj->getShippingPrice() > 0)
                    {
                        $shipping_name = trim(strip_tags($shipping_name));
                        $pk_obj->setUseDelivery(); //for precision correct check
                        $pk_obj->updateFiscalCart(
                            $pk_obj->getPaymentFormType(),
                            $shipping_name,
                            $pk_obj->getShippingPrice(),
                            1,
                            0,
                            $shipping_taxes["tax"]
                        );
                        $pk_obj->delivery_index = count($pk_obj->getFiscalCart()) - 1;
                        $pk_obj->fiscal_cart[$pk_obj->delivery_index]['item_type'] = 'service';
                    }
                }
            }

            if (isset($this->session->data['coupon']) || isset($this->session->data['voucher'])) {
                $set_discount = true;
            }
        }

        if ($this->config->get('handling_status') && $this->config->get('handling_fee') > 0) {
            $tax_rate = $this->getRate($this->config->get('handling_fee'), $this->config->get('handling_tax_class_id'));
            $taxes = $pk_obj->setTaxes($tax_rate);
            $pk_obj->updateFiscalCart(
                $pk_obj->getPaymentFormType(),
                $this->language->get('text_handling'),
                $this->config->get('handling_fee'),
                1,
                0,
                $taxes["tax"]
            );
        }

        if ($this->config->get('paykeeper_force_discounts_check') == '1') {
            $set_discount = true;
        }

        //handle possible discount
        $pk_obj->setDiscounts($set_discount);

        //handle possible precision problem
        $pk_obj->correctPrecision();

        //generate payment form
        $to_hash = number_format($pk_obj->getOrderTotal(), 2, ".", "") .
            $pk_obj->getOrderParams("clientid")     .
            $pk_obj->getOrderParams("orderid")      .
            $pk_obj->getOrderParams("service_name") .
            $pk_obj->getOrderParams("client_email") .
            $pk_obj->getOrderParams("client_phone") .
            $pk_obj->getOrderParams("secret_key");
        $sign = hash ('sha256' , $to_hash);
        return $pk_obj->getDefaultPaymentForm($sign);
    }
}