<?php

// Including the PayKeeper class for payment processing
require_once 'paykeeper.class.php';

/**
 * ControllerPaymentPayKeeper
 *
 * This controller handles the integration with the PayKeeper payment system.
 * It manages the payment initialization, processing, and response handling
 * from PayKeeper.
 *
 * @package     OpenCart
 * @subpackage  Extension
 * @category    Controller
 * @author      support@paykeeper.ru
 * @license     GPL-3.0-or-later
 */
class ControllerPaymentPayKeeper extends Controller {
    /**
     * Handles the initial request for the PayKeeper payment extension and loads the payment view.
     *
     * @return void
     */
    protected function index()
    {
        $this->data['button_confirm'] = $this->language->get('button_confirm');

        $this->load->model('checkout/order');

        $order_info = $this->model_checkout_order->getOrder($this->session->data['order_id']);

        $this->data['sum'] = $this->currency->format($order_info['total'], $order_info['currency_code'], $order_info['currency_value'], false);
        $this->data['orderid'] = $this->session->data['order_id'];
        $this->data['clientid'] = $order_info["firstname"] . " " . $order_info["lastname"];
        $this->data['client_email'] = $order_info['email'];
        $this->data['client_phone'] = $order_info['telephone'];
        $this->data['service_name'] = "";
        $this->data['secret'] = $this->config->get('paykeeper_secret_key');
        $this->data['server'] = $this->config->get('paykeeper_server_url');

        $pk_obj = new PaykeeperPayment();

        $pk_obj->setOrderParams(
            $this->data['sum'],             //sum
            $this->data['clientid'],        //clientid
            $this->data['orderid'],         //orderid
            $this->data['client_email'],    //client_email
            $this->data['client_phone'],    //client_phone
            $this->data['service_name'],    //service_name
            $this->data['server'],          //payment form url
            $this->data['secret']           //secret key
        );

        $this->data["payment_form_type"] = ($pk_obj->getPaymentFormType() == "create") ? "create" : "order";

        // Generate FZ54 fiscal cart (for products, low order fee, shipping)
        $product_info = $this->cart->getProducts();
        $last_index = 0;
        foreach ($product_info as $item) {
            $taxes = array("tax" => "none", "tax_sum" => 0);
            $tax_amount = 0;
            $name = $item["name"];
            $quantity = (float) $item['quantity'];
            if ($item["tax_class_id"]) {
                $tax_rate = $this->getRate($item["price"], $item["tax_class_id"]);
                if (!is_null($tax_rate)) {
                    $taxes = $pk_obj->setTaxes($tax_rate, false);
                    $tax_amount = $item['price'] * ($tax_rate / 100);
                }
            }
            $price = $this->_formatPrice($item["price"] + $tax_amount, $order_info);

            if ($quantity == 1 && $pk_obj->single_item_index < 0) {
                $pk_obj->single_item_index = $last_index;
            }
            if ($quantity > 1 && $pk_obj->more_then_one_item_index < 0) {
                $pk_obj->more_then_one_item_index = $last_index;
            }

            $pk_obj->updateFiscalCart(
                $pk_obj->getPaymentFormType(),
                $name,
                $price,
                $quantity,
                0,
                $taxes["tax"]
            );

            $last_index++;
        }

        // Add shipping details to the fiscal cart
        if (isset($this->session->data["shipping_method"])) {
            $shipping_info = $this->session->data["shipping_method"];
            $shipping_price = $shipping_info['cost'];
            $shipping_taxes = array("tax" => "none", "tax_sum" => 0);
            $shipping_tax_amount = 0;
            if ($shipping_info["tax_class_id"]) {
                $shipping_tax_rate = $this->getRate($shipping_price, $shipping_info["tax_class_id"]);
                if (!is_null($shipping_tax_rate)) {
                    $shipping_taxes = $pk_obj->setTaxes($shipping_tax_rate, false);
                    $shipping_tax_amount = $shipping_price * ($shipping_tax_rate / 100);
                }
            }
            $shipping_price = $this->_formatPrice($shipping_price + $shipping_tax_amount, $order_info);
            if ($shipping_price > 0) {
                $pk_obj->setShippingPrice($shipping_price);
                if (!$pk_obj->checkDeliveryIncluded($pk_obj->getShippingPrice(), $shipping_info['title'])) {
                    $pk_obj->setUseDelivery(); //for precision correct check
                    $pk_obj->updateFiscalCart(
                        $pk_obj->getPaymentFormType(),
                        $shipping_info['title'],
                        $pk_obj->getShippingPrice(),
                        1,
                        0,
                        $shipping_taxes['tax']
                    );
                    $pk_obj->delivery_index = $last_index;
                    $pk_obj->fiscal_cart[$last_index]['item_type'] = 'service';
                }
            }
        }

        // Set discounts
        $pk_obj->setDiscounts(isset($this->session->data['coupon']) || isset($this->session->data['voucher']));

        // Handle any precision issues
        $pk_obj->correctPrecision();

        $this->data["order_form"] = $this->_getPaymentHTMLForm($pk_obj);

        if (file_exists(DIR_TEMPLATE . $this->config->get('config_template') . '/template/payment/paykeeper.tpl')) {
            $this->template = $this->config->get('config_template') . '/template/payment/paykeeper.tpl';
        } else {
            $this->template = 'default/template/payment/paykeeper.tpl';
        }

        $this->render();
    }

    /**
     * Callback method for handling PayKeeper payment confirmation.
     *
     * @return void Outputs success message or error message based on validation checks.
     */
    public function callback()
    {
        // Exit if no POST data is received
        if (!isset($this->request->post)) {
            exit;
        }

        // Check if required POST parameters are present
        if (!isset($this->request->post['id']) || empty($this->request->post['orderid'])){
            echo "Error! Invalid request!";
            exit;
        }

        // Extract and validate key parameters from POST data
        $secret_seed = $this->config->get('paykeeper_secret_key');
        $id = $this->request->post['id'];
        $sum = $this->request->post['sum'];
        $clientid = $this->request->post['clientid'];
        $orderid = $this->request->post['orderid'];
        $key = $this->request->post['key'];

        // Validate hash to ensure data integrity
        $pk_key = md5($id . sprintf("%.2lf", $sum) . $clientid . $orderid . $secret_seed);
        if (!hash_equals($pk_key, $key)) {
            echo "Error! Hash mismatch";
            exit;
        }

        // Retrieve order details and validate the order ID
        $this->load->model('checkout/order');
        $order_info = $this->model_checkout_order->getOrder($orderid);
        if (!isset($order_info['order_id'])) {
            echo "Error! Order was not found";
            exit;
        }

        // Verify that the order total matches the payment amount
        $order_total = $this->_formatPrice($order_info['total'], $order_info);
        $sum_format = number_format($sum, 2, '.', '');
        if ($order_total !== $sum_format) {
            echo "Error! Incorrect order sum!";
            exit;
        }

        // Update order status to reflect successful payment
        if ($order_info['order_status_id'] == 0) {
            $this->model_checkout_order->confirm($orderid, $this->config->get('paykeeper_order_status_id'), 'Payment received by PayKeeper');
        }
        if ($order_info['order_status_id'] != $this->config->get('paykeeper_order_status_id')) {
            $this->model_checkout_order->update($orderid, $this->config->get('paykeeper_order_status_id'), 'Payment received by PayKeeper', true);
        }

        // Output success message with confirmation hash
        echo 'OK ' . md5($id . $secret_seed);
    }

    /**
     * Confirms the order by updating its status.
     *
     * @return void Updates the order history with the pending status.
     */
    public function confirm()
    {
        // Load the order model
        $this->load->model('checkout/order');

        // Load the settings model
        $this->load->model('setting/setting');

        // Получение настроек магазина
        $store_settings = $this->model_setting_setting->getSetting('config', $this->session->data['store_id']);

        // Подтверждение нового заказа
        $this->model_checkout_order->confirm($this->session->data['order_id'], $store_settings['config_order_status_id'], 'Payment received by PayKeeper',TRUE);
    }

    /**
     * Get the tax rate for a given sum and tax class.
     *
     * @param float $sum The sum for which the tax rate needs to be calculated.
     * @param int $tax_class_id The ID of the tax class to determine the appropriate tax rate.
     *
     * @return int Returns the tax rate as an integer. If no tax rate is found, returns 0.
     */
    protected function getRate($sum, $tax_class_id)
    {
        $tax_data = null;
        foreach($this->tax->getRates($sum, $tax_class_id) as $td) {
            $tax_data = $td;
        }
        return (isset($tax_data['rate'])) ? (int) $tax_data['rate'] : null;
    }

    /**
     * Generate the payment form
     *
     * @param $pk_obj
     * @return string
     */
    protected function _getPaymentHTMLForm($pk_obj)
    {
        // Generate the payment form
        $form = "";
        if ($pk_obj->getPaymentFormType() == "create") { //create form
            $to_hash = number_format($pk_obj->getOrderTotal(), 2, ".", "") .
                $pk_obj->getOrderParams("clientid")     .
                $pk_obj->getOrderParams("orderid")      .
                $pk_obj->getOrderParams("service_name") .
                $pk_obj->getOrderParams("client_email") .
                $pk_obj->getOrderParams("client_phone") .
                $pk_obj->getOrderParams("secret_key");
            $sign = hash ('sha256' , $to_hash);

            $form .= '
                <form name="payment" id="pay_form" action="'.$pk_obj->getOrderParams("form_url").'" accept-charset="utf-8" method="post">
                    <input type="hidden" name="sum" value = "'.number_format($pk_obj->getOrderTotal(), 2, ".", "").'"/>
                    <input type="hidden" name="orderid" value = "'.$pk_obj->getOrderParams("orderid").'"/>
                    <input type="hidden" name="clientid" value = "'.$pk_obj->getOrderParams("clientid").'"/>
                    <input type="hidden" name="client_email" value = "'.$pk_obj->getOrderParams("client_email").'"/>
                    <input type="hidden" name="client_phone" value = "'.$pk_obj->getOrderParams("client_phone").'"/>
                    <input type="hidden" name="service_name" value = "'.$pk_obj->getOrderParams("service_name").'"/>
                    <input type="hidden" name="cart" value = \''.htmlentities($pk_obj->getFiscalCartEncoded(),ENT_QUOTES).'\' />
                    <input type="hidden" name="sign" value = "'.$sign.'"/>
                    <input type="submit" id="button-confirm" value="Подтвердить заказ"/>
                </form>
                <script text="javascript">
                    $("#button-confirm").bind("click", function() {
                        $.ajax({ 
                            type: "get",
                            url: "index.php?route=payment/paykeeper/confirm",
                            success: function() {
                                $("#pay_form").submit();
                            }       
                        });
                    });
                </script>';

        }
        else { //order form
            $this->confirm();
            $payment_parameters = array(
                "clientid"=>$pk_obj->getOrderParams("clientid"),
                "orderid"=>$pk_obj->getOrderParams('orderid'),
                "sum"=>number_format($pk_obj->getOrderTotal(), 2, ".", ""),
                "phone"=>$pk_obj->getOrderParams("phone"),
                "client_email"=>$pk_obj->getOrderParams("client_email"),
                "cart"=>$pk_obj->getFiscalCartEncoded()
            );
            $query = http_build_query($payment_parameters);
            $query_options = array("http"=>array(
                "method"=>"POST",
                "header"=>"Content-type: application/x-www-form-urlencoded",
                "content"=>$query
            ));
            $context = stream_context_create($query_options);

            if( function_exists( "curl_init" )) { //using curl
                $CR = curl_init();
                curl_setopt($CR, CURLOPT_URL, $pk_obj->getOrderParams("form_url"));
                curl_setopt($CR, CURLOPT_POST, 1);
                curl_setopt($CR, CURLOPT_FAILONERROR, true);
                curl_setopt($CR, CURLOPT_POSTFIELDS, $query);
                curl_setopt($CR, CURLOPT_RETURNTRANSFER, 1);
                curl_setopt($CR, CURLOPT_SSL_VERIFYPEER, 0);
                $result = curl_exec( $CR );
                $error = curl_error( $CR );
                if( !empty( $error )) {
                    $form .= "<br/><span class=message>"."INTERNAL ERROR:".$error."</span>";
                }
                else {
                    $form .= $result;
                }
                curl_close($CR);
            }
            else { //using file_get_contents
                if (!ini_get('allow_url_fopen')) {
                    $form .= "<br/><span class=message>"."INTERNAL ERROR: Option allow_url_fopen is not set in php.ini"."</span>";
                }
                else {

                    $form .= file_get_contents($pk_obj->getOrderParams("form_url"), false, $context);
                }
            }
        }
        if ($form == "") {
            $form = '<h3>Произошла ошибка при инциализации платежа</h3>';
        }

        return $form;
    }

    /**
     * Formats the given price according to the order's currency settings.
     *
     * @param float $price The price to be formatted.
     * @param array $order_info An associative array containing order details, including:
     *     - 'currency_code' (string): The currency code (e.g., 'USD', 'EUR').
     *     - 'currency_value' (float): The value of the currency relative to the default currency (usually 1.0).
     *
     * @return string The formatted price with two decimal places.
     *
     * @example
     * $order_info = [
     *     'currency_code' => 'USD',
     *     'currency_value' => 1.0
     * ];
     * $formattedPrice = $this->_formatPrice(100.345, $order_info);
     * echo $formattedPrice; // Outputs: "100.35"
     */
    private function _formatPrice($price, $order_info) {
        $price = $this->currency->format(
            floatval($price),
            $order_info['currency_code'],
            $order_info['currency_value'],
            false
        );
        return number_format($price, 2, '.', '');
    }
}
?>
